"""
Defines class Connections that holds data from one or more observations 
(experiments) divided (classified) in groups.

The observations are expected to be generated by scripts celft.py, 
connectors.py, connections.py (depreciated) or classify_connections.py 
(depreciated).  

# Author: Vladan Lucic (Max Planck Institute for Biochemistry)
# $Id$
"""
from __future__ import unicode_literals
from __future__ import absolute_import
from builtins import zip
from builtins import str
from builtins import range

__version__ = "$Revision$"


import warnings
import logging
from copy import copy, deepcopy

import numpy
import scipy

import pyto
from ..util import nested
from .observations import Observations
from .groups import Groups


class Connections(Groups):
    """

    Modes: 
      - 'connectors': Tethers and connectors of the presynaptic terminal
      - 'cleft': Cleft segments
      - 'sv_old': Depreciated version of the presynaptic terminal segmentation

    Attributes set for each group (instance of Observations) belonging to
    this instance for mode 'connectors':
      - ids
      - thresh
      - surface
      - volume
      - length
      - distance
      - boundaryDistance
      - boundaries
    """

    ###############################################################
    #
    # Initialization
    #
    ##############################################################

    def __init__(self, mode=None, single=False):
        """
        Initializes attributes.

        If mode is 'connectors', it is expected tha the data pickles are 
        instances of ..scene.SegmentationAnalysis, that the data describe 
        connectors such as those in the presynaptic terminal, and and that the 
        data was generated by ..scripts.connectors.

        If mode is 'cleft', it is expected tha the data pickles are instances
        of ..scene.SegmentationAnalysis, that the data describe segments
        of the synaptic cleft, and and that the data was generated by 
        ..scripts.cleft.

        If mode is None or 'sv_old', it is expected that the data pickles
        are instances of ..segmentation.Segment, that the data describe 
        connectors (connections) or tethers of the presynaptic terminal,
        and that the data was generated by ..scripts.classify_connections.
        This mode is depreciated.

        """

        # initialize super 
        super(Connections, self).__init__()

        # set mode
        self._mode = mode
        
        # determines the conversion of property names
        self._deep = 'last'

        if (mode is None) or (mode == 'sv_old'):

            # old way, don't do anything else
            self._mode = 'sv_old'

        elif mode == 'connectors':

            # connectors, make property definitions to be used in read() 
            self._full_properties = {
                'ids' : 'ids', 
                'morphology.surface' : 'surface', 
                'morphology.volume' : 'volume', 
                'morphology.length' : 'length',
                'boundDistance.distance' : 'boundaryDistance', 
                'distance.distance' : 'distance'}
            self._full_indexed = [
                'ids', 
                'morphology.surface', 'morphology.volume', 'morphology.length',
                'boundDistance.distance', 'distance.distance']
            self._indexed = [self._full_properties[full_indexed] 
                             for full_indexed in self._full_indexed]

            # adjust for hierarchy and single threshold
            if not single:
                self._full_properties['labels.thresh'] = 'thresh'
                self._full_properties['thresholds'] = 'thresholds'
                self._full_indexed.append('labels.thresh')
            else:
                self._full_properties['labels.threshold'] = 'thresh'
                self._full_indexed.append('labels.threshold')

            self._properties = set(self._full_properties.values())
            self._properties.add('boundaries')
            self._indexed.append('thresh')

        elif mode == 'cleft':

            # cleft, make property definitions to be used in read() 
            self._full_properties = {
                'ids' : 'ids', 
                'density.mean' : 'mean', 'density.std' : 'std', 
                'density.min' : 'min', 'density.max' : 'max', 
                'morphology.surface' : 'surface', 
                'morphology.volume' : 'volume', 'morphology.length' : 'length',
                'topology.euler' : 'euler', 'topology.nLoops' : 'nLoops', 
                'nSegments' : 'nSegments', 'nContacts' : 'nContacts',
                'surfaceDensityContacts' : 'surfaceDensityContacts', 
                'surfaceDensitySegments' : 'surfaceDensitySegments'}
            self._full_indexed = set(
                ['ids',
                 'density.mean', 'density.std', 'density.min', 'density.max', 
                 'morphology.surface', 'morphology.volume', 'morphology.length',
                 'topology.euler', 'topology.nLoops'])
            self._indexed = set(
                [pyto.util.attributes.get_deep_name(attr, mode=self._deep)
                 for attr in self._full_indexed])

            # adjust for hierarchy and single threshold
            if not single:
                self._full_properties['labels.thresh'] = 'thresh'
                self._full_properties['thresholds'] = 'thresholds'
                self._full_indexed = self._full_indexed.union(
                    set(['labels.thresh']))
            else:
                self._full_properties['labels.threshold'] = 'thresh'
                self._full_indexed = self._full_indexed.union(
                    set(['labels.threshold']))

            self._properties = set(self._full_properties.values())
                #[pyto.util.attributes.get_deep_name(attr, mode=self._deep)
                # for attr in self._full_properties])
            self._indexed = self._indexed.union(set(['thresh']))

        else:
            raise ValueError("Mode ", mode, " is not understood. Acceptable ",
                             "values are None, 'connectors', 'cleft', ",
                             "and 'sv_old (depreciated)'.")


    ###############################################################
    #
    # Input
    #
    ##############################################################

    @classmethod
    def read(cls, files, mode=None, single=False, categories=None, order=None, 
             catalog=None, pixel=None):
        """
        Reads values from one or more connection pickles, each pickel 
        corresponding to one experiment. Returns an instance of this
        class that contains data from the pickles read.

        Exact behavior depends on arg mode, which specifies the type of the
        connection pickles, as follows.

        For analysis of pickle files generated by connectors script 
        (typically used for connectors and tethers of the presynaptic terminal 
        generated from 01.2013) use mode connectors and check _readConnectors()
        docstring for more info.

        For analysis of pickle files generated by connections and 
        classify_connections scripts (typically used for connections 
        (connectors) and tethers of the presynaptic terminal generated before 
        12.2012) use mode 'sv_old' and check _readSVOld docstring for further 
        info.

        For analysis of pickle files generated by cleft script (typically used 
        for connectors of the synaptic cleft) use mode 'cleft'
        and check _readCleft() docstring foir further info.

        Argument files has to be a dictionary of dictionaries, where ouside
        keys are group names, inside keys experiment identifiers and 
        inside values file names. For example:

        files = {'group_a' : {'exp_1' : file_1,
                              'exp_2' : file_2,
                              ...             },
                 'group_b' : {'exp_5' : file_5,
                              ...             },
                 ...                           }

        Argument catalog has to be a Catalog object where the groups are 
        already formed (by using Catalog.makeGroups(), for example). That is,
        catalog has to contain the data in attributes that are themselves
        of type dict. For example:

        catalog.pixel_size = {'group_a' : {'exp_1' : pix_size_1,
                                           'exp_2' : pix_size_2,
                                            ...             },
                              'group_b' : {'exp_5' : pix_size_5,
                                            ...             },
                              ...                               }

        Args files and catalog have to have the same groups and observations.

        Arguments:
          - files: dictionary of cleft layers result pickle files
          - mode: type of pickles, currently "connectors", "cleft" or "sv_old"
          - single: Flag indicating if reading single threshold segmentation
          (True) or hierarchical (False)
          - categories: list of categories
          - order: (dictionary of Observations objects) data structure that
         defines the order of the observations (within each group) in this 
         instance
          - catalog: (Catalog) data about experiments, needed for modes 
          'connectors' and 'cleft'
          - pixel: dictionary of pixel sizes, needed for mode 'sv_old'

        Returns instance of this class.
        """

        if (mode is None) or (mode == 'sv_old'):

            conn = cls._readSVOld(files=files, pixel=pixel, 
                                  categories=categories, order=order)

        elif mode == 'cleft':
 
            conn = cls._readCleft(files=files, catalog=catalog, single=single,
                                  categories=categories, order=order)
 
        elif mode == 'connectors':

            conn = cls(mode=mode, single=single)
            conn._readConnectors(
                files=files, catalog=catalog,
                categories=categories, order=order)
 
        else:
            raise ValueError("Mode ", mode, " is not understood. Acceptable ",
                             "values are None, 'connectors', 'cleft', ",
                             "and 'sv_old (depreciated)'.")

        return conn

    @classmethod
    def _readSVOld(cls, files, pixel, mode=None, categories=None, order=None):
        """
        Reads values form connection pickles.

        If ids for an observation is None, all indexed properties (specified 
        in self._indexed) for that observation are set to empty arrays.

        Arguments:
          - files: dictionary of connection result pickle files
          - pixel: dictionary of pixel sizes
          - categories: list of categories
          - order: another (dictionary of Observations objects) data structure
          used to define the order of the identifiers here
          """

        # initialize
        db = pyto.io.Pickled(files)
        conn = cls(mode='sv_old')

        # use all categories if not specified
        if categories is None:
            categories = list(db.categories())

        # loop over categories
        for categ in categories:

            # make sure the identifier order is the same
            if order is not None:
                identifier = order[categ].identifiers
            else:
                identifier = None

            # get data
            conn[categ] = db.readProperties(
                category=categ, identifier=identifier, deep=conn._deep, 
                properties=['ids','morphology.length', 'thresh', 'boundaries',
                            'boundaryDistance', 'distanceToRegion'])
            conn[categ].distance = conn[categ].distanceToRegion

            # set array properties to empty arrays for observations without ids
            for obs_index in range(len(conn[categ].identifiers)):
                if conn[categ].ids[obs_index] is None:
                    conn[categ].ids[obs_index] = numpy.array([])
                    conn[categ].length[obs_index] = numpy.array([])
                    conn[categ].thresh[obs_index] = numpy.array([])
                    conn[categ].boundaryDistance[obs_index] = numpy.array([])
                    conn[categ].boundaries[obs_index] = numpy.array([]) 
                    conn[categ].distance[obs_index] = numpy.array([]) 
                    logging.warning(\
                        "Observation " \
                            + str(conn[categ].identifiers[obs_index])\
                            + " of category " + categ + " has no data.")

            # convert length to nm
            conn[categ].length_nm = \
                conn[categ].pixels2nm(name='length', conversion=pixel[categ])
            conn[categ].distance_nm = \
                conn[categ].pixels2nm(name='distance', conversion=pixel[categ])

            # set book-keeping attributes
            conn[categ].index = 'ids'
            conn[categ].indexed = set(['ids', 'length', 'length_nm', 'thresh',
                      'boundaries'])
            conn[categ].properties = set(['identifiers', 'categories', 'ids',
                       'length', 'length_nm', 'thresh', 'boundaries'])

            #  convert distance to nm
            try:
                conn[categ].distance_nm = \
                    conn[categ].pixels2nm(name='distance', 
                                          conversion=pixel[categ])
                conn[categ].indexed.add('distance')
                conn[categ].indexed.add('distance_nm')
                conn[categ].properties.add('distance')
                conn[categ].properties.add('distance_nm')
            except TypeError:
                if (conn[categ].distance is None) \
                        or any(bd is None for bd \
                                   in conn[categ].distance): 
                    pass
                else:
                    raise

            #  convert boundaryDistance to nm
            try:
                conn[categ].boundaryDistance_nm = \
                    conn[categ].pixels2nm(name='boundaryDistance', 
                                          conversion=pixel[categ])
                conn[categ].indexed.add('boundaryDistance')
                conn[categ].indexed.add('boundaryDistance_nm')
                conn[categ].properties.add('boundaryDistance')
                conn[categ].properties.add('boundaryDistance_nm')
            except TypeError:
                if (conn[categ].boundaryDistance is None) \
                        or any(bd is None for bd \
                                   in conn[categ].boundaryDistance): 
                    pass
                else:
                    raise

        return conn

    @classmethod
    def _readCleft(cls, files, catalog, categories=None, order=None, 
                   single=False):
        """
        Reads ..scene.SegmentationAnalysis pickles specified by arg files and
        puts then in a new instance of this class.

        Each pickle contains data from a single experiment. The properties
        read are specified by attribute self._full_properties.

        In addition, reads other data corresponding to the experiments from 
        arg catalog and puts them together with the data from pickles. The
        only obligatory property is 'pixel_size'.

        If ids for an observation is None, all indexed properties (specified 
        in self._indexed) for that observation are set to empty arrays.

        A category specified by arg categories, or an experiment 
        identifier specified by arg order that does not exist in the data
        (arg files) is ignored and a warning is generated. This condition
        often generates an exception at a later point.

        Arguments:
          - files: dictionary of cleft segmentation result pickle files
          - catalog: (Catalog) data about experiments
          - categories: list of categories
          - order: another Groups instance (or just a dictionary with group 
          names as keys and identifier lists as values), used only to define 
          the order of identifiers in this instance
          - single: Flag indicating if reading single threshold segmentation
          (True) or hierarchical (False)

        Sets properties:
          - ids: connection ids
          - thresh: (indexed) threshold
          - mean, std, min, max: (indexed) grey-scale density 
          - surface, volume, surfaceToVolume: (indexed)
          - euler, nLoops: (indexed) topological properties
          - nSegments: number of connections
          - surfaceToVolume: (indexed) surface to volume ratio
          - surfaceDensity: average (from both cleft boundaries) surface 
          density of connections [1/pix^2]
          - nContacts_1, nContacts_2: (indexed) number of contacts each 
          connection has with cleft boundary 1 and 2 respectivly 
          - surfaceDensityContacts_1/2: surface density of contacts on cleft
          boundaries 1 and 2 respectivly [1/pix^2]
          - thresholds: list of all thresholds
        """

        # initialize
        db = pyto.io.Pickled(files)
        inst = cls(mode='cleft', single=single)

        # use all categories if not specified
        if categories is None:
            categories = list(db.categories())

        # loop over categories
        props_found = {}
        for categ in categories:

            # check if data for the current category exist 
            logging.debug('Connections: Reading group ' + categ)
            if categ not in list(db.categories()):
                logging.warning(
                    'Connections: Data for group ' + categ + ' do not exist')

            # make sure the identifier order is the same
            if order is not None:
                if isinstance(order[categ], Observations):
                    identifier = order[categ].identifiers
                elif isinstance(order[categ], (list, tuple)):
                    identifier = order[categ]
            else:
                identifier = None

            # check if requested identifiers exist in the database
            if identifier is not None:
                clean = []
                for requested in identifier:
                    if requested in db.identifiers():
                        clean.append(requested)
                    else:
                        logging.warning(
                            'Connections: Data for experiment ' + requested + 
                            ' do not exist')
                identifier = clean

            # get data from all experiments of this category
            group = Observations()
            for group, obj, categ_tmp, name_tmp in db.readPropertiesGen(
                category=categ, identifier=identifier, 
                properties=inst._full_properties, deep=inst._deep, index='ids', 
                indexed=inst._full_indexed, multi=group):

                logging.debug('Read data of experiment ' + name_tmp) 

                # do something, perhaps
                pass

            # add data for this category
            inst[categ] = group

            # set array properties to empty arrays for observations without ids
            for obs_index in range(len(inst[categ].identifiers)):
                if inst[categ].ids[obs_index] is None:
                    for name in inst._indexed:
                        value = getattr(inst[categ], name)
                        value[obs_index] = numpy.array([])

            # figure out if some properties were not found
            found = set()
            for name in inst._properties:
                value = getattr(group, name, None)
                if value is None:
                    continue
                if all([x is None for x in value]):
                    continue
                found.add(name)

            # set book-keeping attributes
            inst[categ].index = 'ids'
            inst[categ].indexed = inst._indexed.intersection(found)
            inst[categ].properties = inst._properties.intersection(found)

            # add properties from catalog 
            inst[categ].addCatalog(catalog=catalog)

        # calculate additional data properties
        inst.calculateProperties()

        # convert to nm
        inst.convertToNm(catalog=catalog)

        # check if all groups have the same properties
        last = None
        for categ in categories:
            if last is not None:
                if inst[categ].properties != last:
                    raise ValueError("Groups have different properties")
            last = inst[categ].properties
        inst._indexed.intersection_update(last)
        inst._properties.intersection_update(last)
        
        return inst

    def _readConnectors(self, files, catalog, categories=None, order=None):
        """
        Reads values from one or more connection pickles, each pickel 
        corresponding to one experiment. Returns an instance of this
        class that contains data from the pickles read.

        For analysis of pickle files generated by connectors script. These are 
        typically used for connectors and tethers of the presynaptic terminal 
        generated from 01.2013.

        A category specified by arg categories, or an experiment 
        identifier specified by arg order that does not exist in the data
        (arg files) is ignored and a warning is generated. This condition
        often generates an exception at a later point.

        Arguments:
          - files: dictionary of connectors pickle files
          - catalog: (Catalog) data about experiments
          - categories: list of categories
          - order: another Groups instance (or just a dictionary with group 
          names as keys and identifier lists as values) that defines the 
          order of the observations (within each group) in this instance

        Sets properties:
          - ids: connection ids
          - thresh: (indexed) threshold
          - surface, volume, surface_nm, volume_nm: (indexed) in pixels and nm
          - length, length_nm: (indexed) connection length in pixels and nm
          - boundaryDistance: (indexed) distance between boundaries
          - distance: (indexed) distance to a specified region
          ...
          - thresholds: list of all thresholds
        """

        # initialize
        db = pyto.io.Pickled(files)

        # use all categories if not specified
        if categories is None:
            categories = list(db.categories())

        # loop over categories
        props_found = {}
        for categ in categories:

            # check if data for the current category exist 
            logging.debug('Connections: Reading group ' + categ)
            if categ not in list(db.categories()):
                logging.warning(
                    'Connections: Data for group ' + categ + ' do not exist')

            # make sure the identifier order is the same
            if order is not None:
                if isinstance(order[categ], Observations):
                    identifier = order[categ].identifiers
                elif isinstance(order[categ], (list, tuple)):
                    identifier = order[categ]
            else:
                identifier = None

            # check if requested identifiers exist in the database
            if identifier is not None:
                clean = []
                for requested in identifier:
                    if requested in db.identifiers():
                        clean.append(requested)
                    else:
                        logging.warning(
                            'Connections: Data for experiment ' + requested + 
                            ' do not exist')
                identifier = clean

            # get data from all experiments of this category
            group = Observations()
            for group, obj, categ_tmp, current_identif in db.readPropertiesGen(
                category=categ, identifier=identifier, 
                properties=self._full_properties, deep=self._deep, index='ids', 
                indexed=self._full_indexed, multi=group):

                logging.debug('Read data of experiment ' + current_identif) 

                # calculate additional properties for the current observation
                self.extractProperties(group=group, obj=obj, category=categ_tmp,
                                       identifier=current_identif)

            # add data for this category
            self[categ] = group

            # set array properties to empty arrays for observations without ids
            for obs_index in range(len(self[categ].identifiers)):
                if self[categ].ids[obs_index] is None:
                    for name in self._indexed:
                        value = getattr(self[categ], name)
                        value[obs_index] = numpy.array([])

            # figure out if some properties were not found
            found = set()
            for name in self._properties:
                value = getattr(group, name, None)
                if value is None:
                    continue
                if all([x is None for x in value]):
                    continue
                found.add(name)

            # set book-keeping attributes
            self[categ].index = 'ids'
            self[categ].indexed.update(set(self._indexed).intersection(found))
            self[categ].properties.update(
                set(self._properties).intersection(found))
            if len(set(self._indexed) - found) > 0:
                logging.warning("Properties " + str(set(self._indexed) - found)
                                + " not found.")

            # add properties from catalog 
            self[categ].addCatalog(catalog=catalog)

        # calculate additional data properties
        self.calculateProperties()

        # convert to nm
        self.convertToNm(catalog=catalog)

        # check if all groups have the same properties
        last = None
        for categ in categories:
            if last is not None:
                if self[categ].properties != last:
                    raise ValueError("Groups have different properties")
            last = self[categ].properties
        #self._indexed.intersection_update(last)
        #self._properties.intersection_update(last)
        

    ###############################################################
    #
    # Data modifying methods
    #
    ##############################################################

    def extractProperties(self, group, obj, category, identifier):
        """
        Calculates additional properties for one experiment directly from 
        pickle files.

        The calculated properties are added to the already existing group 
        (arg group).

        Properties calculated depend on the mode. See extractPropertiesCleft() 
        and extractPropertiesConnectors() for details.

        Can be used for modes 'cleft' and 'connectors' only.
        """

        if self._mode == 'cleft':
            self.extractPropertiesCleft(
                group=group, obj=obj, category=category, identifier=identifier)
        elif self._mode == 'connectors':
            self.extractPropertiesConnectors(
                group=group, obj=obj, category=category, identifier=identifier)

    def extractPropertiesConnectors(self, group, obj, category, identifier):
        """
        Calculated properties:
          - boundaries: boundary ids (list of length 2 for each segment)
        """

        # boundaries
        if obj.labels.contacts.compactified:
            obj.labels.contacts.expand()
        bounds = [obj.labels.contacts.findBoundaries(segmentIds=id_, nSegment=1)
                  for id_ in obj.labels.ids]
        bounds = numpy.asarray(bounds)
        group.setValue(property='boundaries', identifier=identifier,
                       value=bounds, indexed=True)

    def extractPropertiesCleft(self, group, obj, category, identifier):
        """
        Currently nothing calculated
        """
        pass

    def convertToNm(self, catalog, categories=None):
        """
        Converts certain properties from pixels to nm. The new values are
        assigned to (new) properties named by adding '_nm' to the corresponding
        original property name.

        Properties set depend on the mode. See convertToNmCleft() and 
        convertToNmConnectors() for details.

        Can be used for modes 'cleft' and 'connectors' only.

        Arguments:
          - catalog: (Catalog) has to contain attribute pixel_size
          - categories: list of categories
        """
        
        if self._mode == 'cleft':
            self.convertToNmCleft(catalog=catalog, categories=categories)
        elif self._mode == 'connectors':
            self.convertToNmConnectors(catalog=catalog, categories=categories)

    def convertToNmCleft(self, catalog, categories=None):
        """
        Converts certain properties from pixels to nm for mode 'cleft'. The new 
        values are assigned to (new) properties named by adding '_nm' to the 
        corresponding original property name.

        Currently converted properties are: length, surface, volume, 
        surfaceDensity, surfaceDensityContacts_1 and _2. 
        """

        if categories is None:
            categories = list(self.keys())

        for categ in categories:
            pixel = catalog.pixel_size

            # convert
            self[categ].length_nm = self[categ].pixels2nm(
                name='length', conversion=pixel[categ])
            self[categ].surface_nm = self[categ].pixels2nm(
                name='surface', power=2, conversion=pixel[categ])
            self[categ].volume_nm = self[categ].pixels2nm(
                name='volume', power=3, conversion=pixel[categ])

            # book keeping
            self[categ].properties.update(['length_nm', 'surface_nm', 
                                          'volume_nm'])
            self[categ].indexed.update(['length_nm']) 

            # surface densities
            for name in ['surfaceDensity', 'surfaceDensityContacts_1',
                         'surfaceDensityContacts_2']:
                
                if name in self[categ].properties:
                    nm_value = self[categ].pixels2nm(name=name, power=-2, 
                                                     conversion=pixel[categ])
                    nm_name = name + '_nm'
                    setattr(self[categ], nm_name, nm_value)
                    self[categ].properties.add(nm_name)
                    if name in self[categ].indexed:
                        self[categ].indexed.add(nm_name)

    def convertToNmConnectors(self, catalog, categories=None):
        """
        Converts certain properties from pixels to nm for mode 'connectors'. 
        The new values are assigned to (new) properties named by adding '_nm' 
        to the corresponding original property name.

        Currently converted properties are: length, distance and 
        boundaryDistance.
        """

        if categories is None:
            categories = list(self.keys())

        for categ in categories:
            pixel = catalog.pixel_size

            # convert length
            try:
                self[categ].length_nm = self[categ].pixels2nm(
                    name='length', conversion=pixel[categ])
                self[categ].properties.update(['length_nm'])
                self[categ].indexed.update(['length_nm']) 
            except TypeError:
                if ((self[categ].length is None) 
                    or any(value is None for value in self[categ].length)): 
                    pass
                else:
                    raise

            # convert distance
            try:
                self[categ].distance_nm = self[categ].pixels2nm(
                    name='distance', conversion=pixel[categ])
                self[categ].properties.update(['distance_nm'])
                self[categ].indexed.update(['distance_nm']) 
            except TypeError:
                if ((self[categ].distance is None) 
                    or any(value is None for value in self[categ].distance)): 
                    pass
                else:
                    raise

            # convert boundaryDistance
            try:
                self[categ].boundaryDistance_nm = self[categ].pixels2nm(
                    name='boundaryDistance', conversion=pixel[categ])
                self[categ].properties.update(['boundaryDistance_nm'])
                self[categ].indexed.update(['boundaryDistance_nm']) 
            except TypeError:
                if ((self[categ].boundaryDistance is None) 
                    or any(value is None for value 
                           in self[categ].boundaryDistance)): 
                    pass
                else:
                    raise
        
    def calculateProperties(self, categories=None):
        """
        Calculates additonal properties from the already existing properties. 

        Properties set depend on the mode. See calculatePropertiesCleft() and 
        calculatePropertiesConnectors() for details.

        Can be used for modes 'cleft' and 'connectors' only.

        Arguments:
          - categories: list of categories
        """
 
        if self._mode == 'cleft':
            self.calculatePropertiesCleft(categories=categories)
        elif self._mode == 'connectors':
            self.calculatePropertiesConnectors(categories=categories)

    def calculatePropertiesCleft(self, categories=None):
        """
        Calculates additonal properties. 

        Sets following new properties to each group (Observations instance)
        contained in this object:
          - surfaceToVolume: (indexed) surface to volume ratio
          - surfaceDensity: average (from both cleft boundaries) surface 
          density of connections [1/pix^2]
          - nContacts_1, nContacts_2: (indexed) number of contacts each 
          connection has with cleft boundary 1 and 2 respectivly 
          - surfaceDensityContacts_1/2: surface density of contacts on cleft
          boundaries 1 and 2 respectivly [1/pix^2]

        Argument:
          - categories: list of group names, in None all groups are used
        """

        if categories is None:
            categories = list(self.keys())

        # surface to ratio
        self.apply(funct=numpy.divide, args=['surface', 'volume'], 
                   name='surfaceToVolume')
        for categ in categories:
            self[categ].properties.add('surfaceToVolume')
            self[categ].indexed.add('surfaceToVolume')

        # segments and contacts
        for categ in categories:
            for ident in self[categ].identifiers:

                # connection surface density
                value = self[categ].getValue(identifier=ident, 
                                             property='surfaceDensitySegments')
                self[categ].setValue(identifier=ident, 
                                     property='surfaceDensity', value=value[0])

                # contacts
                ids = self[categ].getValue(identifier=ident, property='ids')
                n_con = self[categ].getValue(identifier=ident, 
                                             property='nContacts')
                self[categ].setValue(identifier=ident, property='nContacts_1', 
                                     value=n_con[1,ids], indexed=True)
                self[categ].setValue(identifier=ident, property='nContacts_2', 
                                     value=n_con[2,ids], indexed=True)
                con_dens = self[categ].getValue(
                    identifier=ident, property='surfaceDensityContacts')
                self[categ].setValue(
                    identifier=ident, property='surfaceDensityContacts_1',
                    value=con_dens[1])
                self[categ].setValue(
                    identifier=ident, property='surfaceDensityContacts_2',
                    value=con_dens[2])

    def calculatePropertiesConnectors(self, categories=None):
        """
        Calculates additonal properties. 

        Nothing calculed in the moment
        """

        pass

    def removeNoBoundaries(self):
        """
        Removes elements of observations that do not contact any boundary.

        Elements (segments) without boundaries can arise from changing 
        (removing) some of the boundary_ids between two consecutive runs 
        of classify_connection series. 
        """

        # select elements with boundaries
        cond = {}
        for categ in self:

            cond[categ] = []
            for obs_index in range(len(self[categ].identifiers)):

                local_cond = []
                for bound in self[categ].boundaries[obs_index]:

                    if (bound is None) or (len(bound) == 0):
                        local_cond.append(False)
                    else:
                        local_cond.append(True)
                    cond[categ].append(local_cond)

        # extract selected elements
        good = self.extract(condition=cond)
        
        return good

    def removeBoundaries(self, boundary, name='ids'):
        """
        Removes connections that contact specified boundaries.

        All elements (of all indexed properties of all observations contained 
        in this instance) that correspond to elements of property 'boundaries'
        (of this instance) that contain at least one of the boundary ids 
        specified in arg boundary are removed.

        Arguments:
          - boundary: (Groups) have to have the same structure (categories
          and observations) as this instance
          - name: name of the property from boundary that contains ids (ids has
          to be an indexed property)
        """

        # remove boundary ids
        for categ in list(boundary.keys()):

            # get boundary ids for all observations in this category
            b_ids = getattr(boundary[categ], name)

            # find good observation elements
            keep = []
            for obs_ind in range(len(b_ids)):

                # find observation elements that contain specified boundaries
                bad = [numpy.intersect1d(id_pair, b_ids[obs_ind]).any() 
                       for id_pair in self[categ].boundaries[obs_ind]]

                # use observation elements that do not contain specified 
                # boundaries
                keep.append(numpy.logical_not(numpy.asarray(bad)))
             
            # extract good elements 
            self[categ] = self[categ].extractIndexed(condition=keep)
                    
    ###############################################################
    #
    # Extraction methods
    #
    ##############################################################

    def extractByVesicles(self, vesicles, categories=None, other=True):
        """
        Extract and returns an instance of this class that contains only those
        connections that contact vesicles present in arg vesicles. 

        If other is True, an instance of this class containing
        connections that are not extracted is also returned.

        Arguments:
          - vesicles: (Vesicles) vesicles object
          - categories: list of group names, None for all groups
          - other: flag indicating if (in addition to extracted) the remaining 
          non-extracted vesicles are returned
        """

        # get categories
        if categories is None:
            categories = list(self.keys())

        ves_conn = self.__class__()
        ves_non_conn = self.__class__()
        for categ in categories:

            contact = []
            non_contact = []
            for obs_b_ids, ves_ids in zip(
                self[categ].boundaries, vesicles[categ].ids):

                # find ids of contacted vesicles for each connection
                contacted_ves = [numpy.intersect1d(b_ids, ves_ids) 
                                 for b_ids in obs_b_ids]

                # find out which connections contact svs
                cont = [len(one_conn_ves) > 0 for one_conn_ves in contacted_ves]
                contact.append(cont)

                # find ids of non-contacted vesicles for each connection
                non_contacted_ves = [numpy.setdiff1d(ves_ids, b_ids) 
                                     for b_ids in obs_b_ids]

                # find out which connections contact svs
                non_cont = [len(one_conn_ves) > 0 
                            for one_conn_ves in non_contacted_ves]
                non_contact.append(non_cont)

            # extract vesicles
            ves_conn[categ] = self[categ].extract(condition=contact)
            if other:
                ves_non_conn[categ] = self[categ].extract(condition=non_contact)

        # return
        if other:
            return ves_conn, ves_non_conn
        else:
            return ves_conn
        
    def splitByDistance(self, distance, name='distance_nm', 
                          categories=None):
        """
        Returns a list of instances of this class, where each object contains 
        data for connections whose distances (attribute specified by name) fall
        into bins specified by arg distances.

        Lower distance bin limits are inclusive, while the upper are exclusive, 
        except for the upper limit of the last distance bin which is inclusive 
        (like numpy.histogram)

        If distance is a single number a single object is returned. Otherwise,
        if distance is a list of numbers, a list of objects is returned.

        Arguments:
          - distance: list of distances, interpreted as distance bins, or if 
          a single number it is a higher distance limit, while 0 is the lower
          - name: name of the distance attribute
          - categories:
        """

        return self.split(value=distance, name=name, categories=categories)


