#!/usr/bin/env python
"""

Establishes a correlation between two arbitrary systems and correlates 
positions of objects of interest (targets) between these two systems. 

The correlation procedure used in this script is direct, meaning there are no 
intermediary systems. Also, both systems have to have the same dimensionality.

Conceptually, the correlation procedure consists of two steps:
  1) Establish a correlation between the two systems (initial and final), 
that is find a coordinate transformation between the two systems.
  2) Correlate positions of obects of interest from one system to the other.

Marker points are needed to determine the transformation  (step 1). That is, 
one needs to identify one set of features that are visible in both systems.

Requires the following coordinates:
  - marker coordinates in the initial system
  - corresponding marker coordinates in the final system
  - (optional) target coordinates the initial and / or final systems

In this script, coordinates of markers and targets are specified by a file. 
Such a file can be generated by clicking on targets in ImageJ. Other formats
can be used as long as the data is in a table format where rows correspond to 
different markers and targets and the x, y, ... coordinates are specified by 
columns. The calculated transformation parameters and the coordinates of the 
correlated target points are written in a results file. 

Standard usage:

1) Edit this file to enter the desired parameters (coordinates of markers 
and objects of interest, and the transformation type)
2) Import and run this file (in IPython or another python shell):
  >>> import correlation_simple_fileio
  >>> correlation_simple_fileio.main()

Advanced usage:

1) Edit this file to enter the desired parameters (coordinates of markers 
and objects of interest, and the transformation type)
2) Import this file and execute commands from main() one by one. Check values 
of variables as needed:
  >>> import correlation_simple_fileio
  >>> from correlation_simple_fileio import *
  >>> markers = {
        'markers_1' : (coordinates_file, marker_initial_rows),
        'markers_2' : (coordinates_file, marker_final_rows)}
  >>> corr = Basic()
  >>> corr.readPositions(
        points=markers, format_=file_format, comments=comments, 
        skiprows=skiprows, delimiter=delimiter, columns=columns, 
        indexing=indexing)
  >>> corr.establish(type_=transform_type)
      ...
  >>> corr.transf_1_to_2.phiDeg
  >>> corr.transf_1_to_2.gl
      ...
Other attributes of corr are listed in the documentation for 
pyto.correlative.Basic. If the initial and final systems are 2D, object 
corr.transf_1_to_2 is an instance of pyto.geometry.Affine2D, if they are 3D
corr.transf_1_to_2 is an instance of pyto.geometry.AffinewD and in other cases
of pyto.geometry.Affine. Please check docs for these classes for other 
attributes. 

If you use this script, please consider citing: 

  Fukuda, Y., N. Schrod, M. Schaffer, L. R. Feng, W. Baumeister, and V. Lucic, 
  2014. Coordinate transformation based cryo-correlative methods for electron 
  tomography and focused ion beam milling. Ultramicroscopy 143:15– 23. 


# Author: Vladan Lucic (Max Planck Institute for Biochemistry)
# $Id$
"""
from __future__ import unicode_literals
#from builtins import str
from builtins import range

__version__ = "$Revision$"

import sys
import os
import time
import platform

import pyto
from pyto.correlative.basic import Basic


##################################################################
#
# Parameters
#
#################################################################

##################################################################
#
# Marker and target coordinates
#

# file containing marker and target coordinates in the initial system
coordinates_file = 'correlation_data.dat'

# rows containing marker coordinates in the initial system
marker_initial_rows = [7, 8, 9, 10]

# rows containing targets in the initial system
marker_final_rows = [1,2,3,4]

# rows containing target coordinates in the initial system
target_initial_rows = [5, 6]

# rows containing targets in the initial system
target_final_rows = [11, 12, 13]

# transformation type: 'gl' for general linear or 'rs' for rigid
transform_type = 'gl'

##################################################################
#
# Results output
#

# result output file name
results_file = 'correlation_simple_fileio_results.dat'


##################################################################
#
# Coordinates file format
#

# coordinates file format
file_format = 'imagej'

# count rows and columns from this number
indexing = 1

# columns that contain coordinates 
columns = [3, 4]

# comments symbol (not used if file format 'imagej')
comments = '#'

# number of rows to skip from the top (not used if file format is 'imagej')
skiprows = 1

# field delimiter (not used if file format 'imagej')
delimiter='\t'


#####################################################################
#
# Functions
#
#####################################################################

def machine_info():
    """
    Returns machine name and machine architecture strings
    """
    mach = platform.uname() 
    mach_name = mach[1]
    mach_arch = str([mach[0], mach[4], mach[5]])

    return mach_name, mach_arch

def write_results(corr, res_file_name):
    """
    Writes results to a file
    """

    # open results file
    res_file = open(res_file_name, 'w')
    
    # machine info
    mach_name, mach_arch = machine_info()
    header = ["#",
        "# Machine: " + mach_name + " " + mach_arch,
        "# Date: " + time.asctime(time.localtime())]
    
    # file names and times
    script_file_name = sys.modules[__name__].__file__
    script_time = time.asctime(
        time.localtime(os.path.getmtime(script_file_name)))
    header.extend([
        "#",
        "# Input script: " + script_file_name + " (" + script_time + ") " 
        + __version__,
        "# Working directory: " + os.getcwd()])
            
    # transformation parameters
    header.extend([
        "#",
        "# Transformation parameters:",
        "#",
        "#   - rotation = %6.1f" % corr.transf_1_to_2.phiDeg,
        "#   - scale = [%6.3f, %6.3f]" 
                % (corr.transf_1_to_2.scale[0], corr.transf_1_to_2.scale[1]), 
        "#   - parity = %d" % corr.transf_1_to_2.parity, 
        "#   - shear = %7.3f" % corr.transf_1_to_2.shear,
        "#   - translation = [%6.3f, %6.3f]" % (corr.transf_1_to_2.d[0],
                                                corr.transf_1_to_2.d[1]),
        "#   - rms error: ", 
        "#       - in final system units %6.2f" % corr.transf_1_to_2.rmsError, 
        "#       - in initial system units %6.2f" % corr.transf_2_to_1.rmsError
    ])
    header.extend([
        "#   - error (in EM overview units): "])
    for error_line in corr.transf_1_to_2.error:
        header.append(
            "#        " + str(error_line))

    # write header
    for line in header:
        res_file.write(line + os.linesep)

    # correlation results initial to final
    table = []
    try:
        if corr.correlated_1_to_2 is not None:
            table.extend([
                "#",
                "# Correlation of targets from initial to final system",
                "#",
                "#  Id      Initial          Final   " ])
            out_vars = [
                corr.targets_1[:,0], corr.targets_1[:,1],
                corr.correlated_1_to_2[:,0], corr.correlated_1_to_2[:,1]]
            out_format = ' %3u   %6.2f %6.2f   %6.2f %6.2f  '
            n_res = corr.targets_1.shape[0]
            ids = list(range(n_res))
            res_tab = pyto.io.util.arrayFormat(
                arrays=out_vars, format=out_format, indices=ids, 
                prependIndex=True)
            table.extend(res_tab)
    except AttributeError:
        pass

    # correlation results final to initial
    try:
        if corr.correlated_2_to_1 is not None:
            table.extend([
                "#",
                "# Correlation of targets from final to initial system",
                "#",
                "#  Id      Initial          Final   " ])
            out_vars = [
                corr.correlated_2_to_1[:,0], corr.correlated_2_to_1[:,1],
                corr.targets_2[:,0], corr.targets_2[:,1]]
            out_format = ' %3u   %6.2f %6.2f   %6.2f %6.2f  '
            n_res = corr.targets_2.shape[0]
            ids = list(range(n_res))
            res_tab = pyto.io.util.arrayFormat(
                arrays=out_vars, format=out_format, indices=ids, 
                prependIndex=True)
            table.extend(res_tab)
    except AttributeError:
        pass

    # write data table
    for line in table:
        res_file.write(line + os.linesep)

            
#####################################################################
#
# Main function
#
#####################################################################

def main():
    """
    Main function
    """

    # make marker coordinates dictionary
    markers = {
        'markers_1' : (coordinates_file, marker_initial_rows),
        'markers_2' : (coordinates_file, marker_final_rows)
        }

    # establish correlation
    corr = Basic()
    corr.readPositions(
        points=markers, format_=file_format, comments=comments, 
        skiprows=skiprows, delimiter=delimiter, columns=columns, 
        indexing=indexing)
    corr.establish(type_=transform_type)

    # corrtale target coordinates
    targets = {
        'targets_1' : (coordinates_file, target_initial_rows),
        'targets_2' : (coordinates_file, target_final_rows)
        }
    corr.readPositions(
        points=targets, format_=file_format, comments=comments, 
        skiprows=skiprows, delimiter=delimiter, columns=columns,
        indexing=indexing)
    corr.correlate()

    # write results
    write_results(corr=corr, res_file_name=results_file)


# run if standalone
if __name__ == '__main__':
    main()

