"""
Generates some unusual structuring elements.

# Author: Vladan Lucic (Max Planck Institute for Biochemistry)
# $Id$
"""
from __future__ import unicode_literals
from __future__ import division
from builtins import str
from builtins import range
from builtins import object
from past.utils import old_div

__version__ = "$Revision$"

import logging
import numpy
import scipy
import scipy.ndimage as ndimage

class StructEl(object):
    """
    Defines some unusual structuring elements and provides label method that 
    uses these structure.

    Currently implemented (in addition those of size 3, generated by
    scipy.ndimage.generate_binary_structure method):
      - connectivity = 4, size = 5
      - linear (cross-like in 2d), size 5
      - higher connectivity and sizes, but labeling might generate artefacts

    Important attributes:
      - self.rank: ndim of the structuring element (as in
      scipy.ndimage.generate_binary_structure)
      - self.size: size of the structuring element (same in each dimension)
      - self.mode: defines structuring element type. Can be:
        - None (or 'standard'):
        - 'linear'
        - '1d'
    - self.connectivity: square of the distance from the center that 
      defines which elements belong to the structuring element (as in
      scipy.ndimage.generate_binary_structure). Used only in the standard mode.
      - self.axis: axis along which the structuring element has 1's, used only 
      for the linear mode.
    """

    def __init__(
            self, rank=None, mode=None, connectivity=None, size=None, 
            axis=None):
        """
        Initializes attributes.

        If size is None, but connectivity is specified, the size is set to the
        minimal value consistent with the connectivity. That is, for 
        connectivity 1, 2 or 3 size is 3, and for connectivity 4 size is 5.
        """

        # parse arguments
        self.rank = rank
        self.connectivity = connectivity
        self.size = size
        self.mode = mode
        self.axis = axis

        # if size is None, set minimal size according to the connectivity
        if (self.size is None) and (self.connectivity is not None):
            half_size = int(numpy.floor(numpy.sqrt(self.connectivity)))
            self.size = (2 * half_size) + 1

    ##################################################################
    #
    # Methods
    #

    @classmethod
    def label(cls, input, structure=None):
        """
        Labels a (binary) image. Extends scipy.ndimage.label so that
        structuring elements of sizes greater than 3 can be used.

        Labeling is decomposed into connecting (unconnected) segments of image
        using an appropriate structuring element, followed by labeling with (an
        appropriate) size 3 structuring element. See self.decompose for further
        info abnout this decomposition.

        Arguments:
          - input: image to be labeled
          - structure: structuring element, an instance of StructEl or an array
          of size 3

        Returns: image, n_labels
          - image: labeled image
          - n: number of labels
         """

        # warn
        if structure.size > 5:
            logging.warning(
                "Labeling with a structuring element whose size > 5 is " +
                " not recommended beacause it mat introduce artefacts.")
        if structure.connectivity > 4:
            logging.warning(
                "Labeling with a structuring element whose connectivity " +
                "> 5 is not recommended because most likely it " +
                "introduces artefacts.")

        if isinstance(structure, cls):

            # decompose structure
            label_se, conn_se = structure.decompose()

            # connect input using connect structuring element
            if conn_se is not None:
                conn_input = conn_se.connect(input)
            else:
                conn_input = input

            # label connected input using labeling structuring element
            label_se_array =  label_se.generate()
            conn_label, nn = ndimage.label(conn_input, structure=label_se_array)

            # restrict labels to input foreground and return
            conn_label[input==0] = 0
            return conn_label, nn

            # complain
            raise NotImplementedError(
                "Sorry, don't know how to label using " +
                "structuring element with " +
                "rank: " + str(structure.rank) +
                ", mode: " + str(structure.mode) +
                ", connectivity: " + str(structure.connectivity) +
                " and size: " + str(structure.size) + ".")

        else:

            # presumably structure is a ndarray
            # ToDo: if structure size > 3 try to decompose
            return ndimage.label(input, structure=structure)

    def decompose(self):
        """
        Decomposes structuring elements with size > 3 into a structuring element
        that can be used to connect disconnected pieces (see self.conect) and a
        structuring element used for labeling.

        Examples:
          1) Labeling with structuring element of size 5 and connectivity 4
          returns both connection and labeling structuring element of size 3 
          and connectivity 1.
          2) Labeling with a structuring element that has entries on central 
          axes only (cross in 2d) returns size 3 and connectivity 1 labeling 
          structuring element and 1d size 3 connection structuring element 
          that is applied along all axes.

        Returns: se, connect:
          - label_se: labeling structuring element
          - connect_se: connect structuring element
        """

        if (self.size is None) or (self.size==3):

            # size 3, no decomposition needed
            return self, None

        elif (self.size > 3) and (self.size%2 == 1):
            if (self.mode is None) or (self.mode == 'standard'):

                # size > 3, standard
                half_size = old_div((self.size - 1), 2)
                if self.connectivity < half_size**2:

                    # size too large: trim the size and call decompose
                    se = self.__class__(
                        rank=self.rank, size=self.size-2,
                        mode=self.mode, connectivity=self.connectivity)
                    return se.decompose()
                    
                elif self.connectivity == half_size**2:

                    # decomposes into basic se
                    label_se = self.__class__(rank=self.rank, size=3, mode=None,
                                              connectivity=3)
                    new_conn = (numpy.sqrt(self.connectivity) - 1)**2
                    conn_se = self.__class__(rank=self.rank, size=self.size-2,
                                             connectivity=new_conn, mode=None)
                    return label_se, conn_se

                else:

                    # ToDo, might be needed
                    pass

            elif self.mode == 'linear':

                if self.axis is None:

                    # size > 3, linear, allong all axes
                    label_se = self.__class__(rank=self.rank, size=3, mode=None,
                                              connectivity=1)
                    conn_se = self.__class__(rank=self.rank, mode='1d',
                                             size=self.size-2, axis=None)
                    return label_se, conn_se
       
        # complain
        raise NotImplementedError(
            "Sorry, don't know how to decompose " +
            "structuring element with " +
            "rank: " + str(self.rank) +
            ", mode: " + str(self.mode) +
            ", connectivity: " + str(self.connectivity) +
            " and size: " + str(self.size) + ".")
        
    def connect(self, input):
        """
        Connects parts of input array using this structuring element.

        Done by adding elements or array that 
        """

        if ((self.mode is None) or (self.mode == 'standard') 
            or (self.mode == 'linear')):

            # standard or linear structuring element: generate and correlate
            se = self.generate()
            corr = ndimage.correlate(input, se, mode='constant')
            new = (corr > 1) | (input > 0)
            return new

        elif self.mode == '1d':
            if self.axis is None:
            
                # apply 1d structuring element along each axis
                new = (input > 0)
                se = self.generate()
                for axis in range(input.ndim):
                    corr = ndimage.correlate1d(
                        input, se, axis=axis, mode='constant')
                    new = new | (corr > 1)
                return new
                
            else:

                # 1d structuring element along a given axis
                se = self.generate()
                corr = ndimage.correlate1d(
                    input, se, axis=self.axis, mode='constant')
                new = (corr > 1) | (input>0)
                return new

        # complain
        raise NotImplementedError(
            "Sorry, don't know how to connect using " +
            "structuring element with " +
            "rank: " + str(self.rank) +
            ", mode: " + str(self.mode) +
            ", connectivity: " + str(self.connectivity) +
            " and size: " + str(self.size) + ".")

    def generate(self):
        """
        Generates a structuring element (array) based on the attributes of 
        this instance.

        Returns a structuring element as an array.
        """

        if (self.mode is None) or (self.mode == 'standard'):
            if self.size == 3:

                # size 3, standard
                return ndimage.generate_binary_structure(
                    self.rank, self.connectivity)

            elif (self.size > 3) and (self.size%2 == 1):

                # size >3, standard
                shape = [self.size] * self.rank
                distance_in = numpy.ones(shape, dtype='int')
                origin = (old_div((self.size-1),2),) * self.rank
                distance_in[origin] = 0
                if (distance_in > 0).all():  # workaround for scipy bug 1089
                    raise ValueError("Can't calculate distance_function ",
                                     "(no background)")
                else:
                    distance = ndimage.distance_transform_edt(distance_in)
                se = numpy.where(
                    distance > numpy.sqrt(self.connectivity), False, True)
                return se

        elif self.mode == 'linear':

            # linear, self.rank-dimensional
            shape = [self.size] * self.rank
            se = numpy.zeros(shape, dtype='bool')
            if self.axis is None:
                for axis in range(self.rank):
                    true_indices = [slice(None)] * self.rank
                    true_indices[axis] = old_div((self.size-1), 2)
                    se[tuple(true_indices)] = True
            else:
                true_indices = [slice(None)] * self.rank
                true_indices[self.axis] = old_div((self.size-1), 2)
                se[tuple(true_indices)] = True
            return se
                
        elif self.mode == '1d':

            # linear, 1-dimensional
            return numpy.ones(shape=self.size, dtype='bool')


        # complain
        raise NotImplementedError(
            "Sorry, don't know how to generate a " +
            "structuring element with " +
            "rank: " + str(self.rank) +
            ", mode: " + str(self.mode) +
            ", connectivity: " + str(self.connectivity) +
            " and size: " + str(self.size) + ".")

        
