function tomolist = tm_imod_preprocess(tomolist, p, imodpp, write_list)
%% tm_imod_preprocess
% A function for taking a tomolist and running batched initial IMOD 
% preprocessing.
%
% WW 12-2017


%% Initialize

% Number of stacks
n_stacks = size(tomolist,1);

% When to end batchruntomo
endnum=0;

%% Write directive file and preprocess for each tomogram

for i = 1:n_stacks
     
    % Check processing
    process = true;
    if tomolist(i).skip
        process = false;        
    elseif tomolist(i).imod_preprocessed
        if ~imodpp.force_imod
            process = false;
        end
    end
        
    
    
    % Perform IMOD preprocessing
    if process        
        disp([p.name,'Initializing batck processing for stack ',tomolist(i).stack_name]);
        
        % Parse stack name
        switch imodpp.process_stack
            case 'unfiltered'
                stack_name = tomolist(i).stack_name;
            case 'dose-filtered'
                stack_name = tomolist(i).dose_filtered_stack_name;
            otherwise
                error([p.name,'ACHTUNG!!! ',imodpp.process_stack,' is an unsupported stack type!!! Allowed types are either "unfiltered" or "dose-filtered"']);
        end
        [~,name,ext] = fileparts(stack_name);

        % Initialize IMOD directory
        imod_dir = [tomolist(i).stack_dir,'imod/'];
        if exist(imod_dir,'dir')
           system(['rm -rf ',imod_dir]); 
        end
        mkdir(imod_dir);
        system(['ln -s ',tomolist(i).stack_dir,stack_name,' ',imod_dir]);
        system(['cp ',tomolist(i).stack_dir,name,'.rawtlt ',imod_dir]);
        
        
        % Check for old, unprocessed stack
        if imodpp.force_imod
            
            % Check for archive
            disp([p.name,'Forced repeat preprocessing... Checking for archived stack...']);
            if exist([imod_dir,name,'_xray',ext,'.gz'],'file')
                disp([p.name,'Compressed archived stack found. Unarchiving stack...']);
                system(['archiveorig -r ',imod_dir,stack_name]);
            end
            
            % Check for _orig file
            if exist([imod_dir,name,'_orig',ext],'file')
                disp([p.name,'Forced repeat preprocessing... Original tilt stack ',name,'_orig',ext,' found and will be used...']);
                movefile([imod_dir,name,'_orig',ext], [imod_dir,stack_name]);
            end
                        
        end        
        
        % Initialize adoc ouptut
        adoc_name = [imod_dir,'tomoman_preprocess.adoc'];
        adoc = fopen(adoc_name,'w');
        fprintf(adoc,'# An adoc file for preprocessing generated by tomoman_imod_preprocess \n\n');
        fprintf(adoc,['setupset.datasetDirectory=',imod_dir,'\n']);

        
        % Copytomocoms
        if imodpp.copytomocoms
            fprintf(adoc,['setupset.copyarg.name=',name,'\n']);
            fprintf(adoc,['setupset.copyarg.dual=0','\n']);
            fprintf(adoc,['setupset.copyarg.pixel=',num2str(tomolist(i).pixelsize/10),'\n']);
            fprintf(adoc,['setupset.copyarg.gold=',num2str(imodpp.goldsize),'\n']);
            if ~isempty(imodpp.rotation)
                fprintf(adoc,['setupset.copyarg.rotation=',num2str(imodpp.rotation),'\n']);
            else
                fprintf(adoc,['setupset.copyarg.rotation=',num2str(tomolist(i).tilt_axis_angle),'\n']);
            end
            fprintf(adoc,'setupset.copyarg.userawtlt=1\n');                        
        end

        % CCD Eraser
        if imodpp.ccderaser
            fprintf(adoc,'runtime.Preprocessing.any.removeXrays=1\n');
            if imodpp.archiveoriginal == 1
                fprintf(adoc,'runtime.Preprocessing.any.archiveOriginal=1\n');
            end
            endnum=1;
        end

        % Coarse alignment (Added coarse alignment binning support 08-2020)
        if imodpp.coarsealign
            if ~isempty(imodpp.tiltxcorrbinning)
                fprintf(adoc,['comparam.xcorr.tiltxcorr.BinningToApply=',num2str(imodpp.tiltxcorrbinning),'\n']);
            end
            if ~isempty(imodpp.tiltxcorrangleoffset)
                fprintf(adoc,['comparam.xcorr.tiltxcorr.AngleOffset=',num2str(imodpp.tiltxcorrangleoffset),'\n']);
            end            
            if imodpp.ExcludeCentralPeak
                fprintf(adoc,'comparam.xcorr.tiltxcorr.ExcludeCentralPeak\n');
            end
            if ~isempty(imodpp.ShiftLimitsXandY)
                fprintf(adoc,['comparam.xcorr.tiltxcorr.ShiftLimitsXandY=',num2str(imodpp.ShiftLimitsXandY(1)),',',num2str(imodpp.ShiftLimitsXandY(2)),'\n']);
            end
            fprintf(adoc,['comparam.prenewst.newstack.BinByFactor=',num2str(imodpp.coarsealignbin),'\n']);
            fprintf(adoc,['comparam.prenewst.newstack.AntialiasFilter=',num2str(imodpp.coarseantialias),'\n']);
            fprintf(adoc,['comparam.prenewst.newstack.ModeToOutput=',imodpp.convbyte,'\n']);
            endnum=3;
        end
        
        % Tracking and Seeding
        
        switch imodpp.tracking_method
            
            case -1
                disp([p.name,'skipping automated tracking...']);
                
            case 0
                disp([p.name,'Using automated fiducial tracking...']);
                
%                 % Autoseed
%                 fprintf(adoc,'runtime.Fiducials.any.trackingMethod=0\n');
%                 fprintf(adoc,'runtime.Fiducials.any.seedingMethod=1\n');
%                 fprintf(adoc,['comparam.autofidseed.autofidseed.TwoSurfaces=',num2str(imodpp.two_surf),'\n']);
%                 fprintf(adoc,['comparam.autofidseed.autofidseed.TargetNumberOfBeads=',num2str(imodpp.n_beads),'\n']);
%                 fprintf(adoc,['comparam.autofidseed.autofidseed.AdjustSizes=',num2str(imodpp.adjustsize),'\n']);
%                 fprintf(adoc,['comparam.track.beadtrack.LocalAreaTracking=',num2str(imodpp.localareatracking),'\n']);
%                 fprintf(adoc,['comparam.track.beadtrack.LocalAreaTargetSize=',num2str(imodpp.localareasize),'\n']);
%                 fprintf(adoc,['comparam.track.beadtrack.SobelFilterCentering=',num2str(imodpp.sobelfilter),'\n']);
%                 fprintf(adoc,['comparam.track.beadtrack.KernelSigmaForSobel=',num2str(imodpp.sobelkernel),'\n']);
%                 fprintf(adoc,'comparam.track.beadtrack.ScalableSigmaForSobel=\n');  % Make sure this is disabled; can sometimes cause problems
%                 fprintf(adoc,['comparam.track.beadtrack.RoundsOfTracking=',num2str(imodpp.n_rounds),'\n']);
%                 fprintf(adoc,['runtime.BeadTracking.any.numberOfRuns=',num2str(imodpp.n_runs),'\n']);
%                 endnum=5;
                
                % Autoseed
                fprintf(adoc,'runtime.Fiducials.any.trackingMethod=0\n');
                fprintf(adoc,'runtime.Fiducials.any.seedingMethod=1\n');
                fprintf(adoc,['comparam.track.beadtrack.LocalAreaTracking=',num2str(imodpp.localareatracking),'\n']);
                fprintf(adoc,['comparam.track.beadtrack.LocalAreaTargetSize=',num2str(imodpp.localareasize),'\n']);
                fprintf(adoc,['comparam.track.beadtrack.SobelFilterCentering=',num2str(imodpp.sobelfilter),'\n']);
                fprintf(adoc,['comparam.track.beadtrack.KernelSigmaForSobel=',num2str(imodpp.sobelkernel),'\n']);
                fprintf(adoc,'comparam.track.beadtrack.ScalableSigmaForSobel=\n');  % Make sure this is disabled; can sometimes cause problems
                fprintf(adoc,['comparam.track.beadtrack.RoundsOfTracking=',num2str(imodpp.n_rounds),'\n']);
                fprintf(adoc,['runtime.BeadTracking.any.numberOfRuns=',num2str(imodpp.n_runs),'\n']);
                fprintf(adoc,['comparam.autofidseed.autofidseed.TwoSurfaces=',num2str(imodpp.two_surf),'\n']);
                fprintf(adoc,['comparam.autofidseed.autofidseed.TargetNumberOfBeads=',num2str(imodpp.n_beads),'\n']);
                fprintf(adoc,['comparam.autofidseed.autofidseed.AdjustSizes=',num2str(imodpp.adjustsize),'\n']);
                endnum=5;
                
            case 1
                disp([p.name,'Using automated patch tracking...']);
                
                % Patch tracking
                fprintf(adoc,'runtime.Fiducials.any.trackingMethod=1\n');
                fprintf(adoc,['comparam.xcorr_pt.tiltxcorr.SizeOfPatchesXandY=',num2str(imodpp.SizeOfPatchesXandY,'%i,%i'),'\n']);
                if isfield(imodpp,'NumberOfPatchesXandY')       % There can only be one...
                    fprintf(adoc,['comparam.xcorr_pt.tiltxcorr.NumberOfPatchesXandY=',num2str(imodpp.NumberOfPatchesXandY,'%i,%i'),'\n']);
                elseif isfield(imodpp,'OverlapOfPatchesXandY')
                    fprintf(adoc,['comparam.xcorr_pt.tiltxcorr.OverlapOfPatchesXandY=',num2str(imodpp.OverlapOfPatchesXandY,'%5.2f,%-5.2f'),'\n']);
                end
                fprintf(adoc,['comparam.xcorr_pt.tiltxcorr.IterateCorrelations=',num2str(imodpp.IterateCorrelations),'\n']);
                endnum=4;
        end
        
        
%         % other parameters (SAGAR 08-2020)
%         if imodpp.position_tomo
%             fprintf(adoc,'runtime.Positioning.any.sampleType=2\n');
%             fprintf(adoc,'runtime.Positioning.any.wholeTomogram=1\n');
%             fprintf(adoc,['runtime.Positioning.any.binByFactor=',num2str(imodpp.positioning_binning),'\n']); % add option
%             fprintf(adoc,['runtime.Positioning.any.thickness=',num2str(imodpp.positioning_thickness),'\n']);
%             fprintf(adoc,['runtime.AlignedStack.any.binByFactor=',num2str(imodpp.alignedstack_binning),'\n']);
%         end
        
        fclose(adoc);
    
        % Run batchtomorun
        disp([p.name,'Running IMOD preprocessing...']);
        system(['batchruntomo -directive ',adoc_name,' -end ',num2str(endnum)]);
        disp([p.name,'IMOD preprocessing on stack ',tomolist(i).dose_filtered_stack_name, ' complete!!!!']);
        
        % Update tomolist
        tomolist(i).imod_preprocessed = true;
        tomolist(i).alignment_stack = imodpp.process_stack;
        tomolist(i).alignment_software = 'imod';
        
        % Save tomolist
        if write_list
            save([p.root_dir,p.tomolist_name],'tomolist');
    
        end

    end 
end


