"""
Defines class Clusters that holds clusters-related data from one or more observations 
(experiments) divided (classified) in groups.

The observations are expected to be generated by scripts/cluster_analysis.py.  

# Author: Vladan Lucic (Max Planck Institute for Biochemistry)
# $Id$
"""
from __future__ import unicode_literals
from __future__ import absolute_import
from __future__ import division
from builtins import zip
from builtins import range
#from past.utils import old_div

__version__ = "$Revision$"


import warnings
import logging
import os.path
from copy import copy, deepcopy

import numpy
import scipy

import pyto
from ..util import nested
from .groups import Groups
from .observations import Observations


class Clusters(Groups):
    """
    """

    ###############################################################
    #
    # Initialization
    #
    ##############################################################

    def __init__(self):
        """
        Initializes attributes
        """

        # initialize super 
        super(Clusters, self).__init__()

    ###############################################################
    #
    # Input
    #
    ##############################################################

    @classmethod
    def read(cls, files, mode, catalog=None, categories=None, order=None, 
             distances=None):
        """
        Reads values form cluster pickles.

        If distances are not None the boundary distances are also read. This 
        argument can be specified in the same form as arg files, or set to 
        'default' in which case the distance pickle names are derived from 
        cluster pickle names (substitute '_cluster.pkl' by 
        '_bound_distances.pkl').

        Argument files has to be a dictionary of dictionaries, where ouside
        keys are group names, inside keys experiment identifiers and 
        inside values file names. For example:

        files = {'group_a' : {'exp_1' : file_1,
                              'exp_2' : file_2,
                              ...             },
                 'group_b' : {'exp_5' : file_5,
                              ...             },
                 ...                           }

        It is recommended that the data about experiments is specified by
        arg catalog. Catalog has to be a Catalog object where the groups are 
        already formed (by using Catalog.makeGroups(), for example). That is,
        catalog has to contain the data in attributes that are themselves
        of type dict. For example:

        catalog.pixel_size = {'group_a' : {'exp_1' : pix_size_1,
                                           'exp_2' : pix_size_2,
                                            ...             },
                              'group_b' : {'exp_5' : pix_size_5,
                                            ...             },
                              ...                               }

        Args files and catalog have to have the same groups and observations.

        A category specified by arg categories, or an experiment 
        identifier specified by arg order that does not exist in the data
        (arg files) is ignored and a warning is generated. This condition
        often generates an exception at a later point.

       Arguments:
          - files: dictionary of cluster pickle files
          - catalog: (Catalog) data about experiments (
          - categories: list of categories
          - mode: clustering mode: 'connectivity' (or 'conn'), 
          'hierarchicalBoundaries' (or 'hiBound'), or 'hierarchicalConnections' 
          (or 'hiConn')
          - order: another Groups instance (or just a dictionary with group 
          names as keys and identifier lists as values that defines the 
          order of the identifiers here
          - distances: dictionary of boundary distances pickle files, or
          'default' to derive distance pickle names from cluster pickle names

        Sets properties:
          - 'ids' (indexed): (list of ndarrays) cluster ids
          - 'identifiers' 
          - 'categories'
          - 'n': (list) number of clusters in each observation
          - 'bound_clusters' (indexed): (list of ndarrays of sets) boundary ids
          arranged by (boundary) clusters
          - 'conn_clusters' (indexed): (list of ndarrays of sets) connection ids
          arranged by (connection) clusters
          - 'bound_ids (not indexed): (list of ndarrays) all boundary ids in 
          the order corresponding to distances   
          - 'bound_dist': (list of ndarrays) pairwise distances between 
          boundaries in the vector form (as returned by 
          scipy.spatial.distance.pdist) according to the order given in 
          bound_ids
          - 'n_connections', 'n_links' (indexed): (list of ndarrays) number of
          connections, links
          - 'euler', 'euler_links' (indexed): (list of ndarrays) Euler 
          characteristics based on connections, links
          - 'loops', 'loops_links' (indexed): (list of ndarrays) number of 
          independent loops based on connections, links
          - 'branches', 'branches_links' (indexed): (list of ndarrays) number of
          branches based on connections, links
          - 'distances' and 'distances_nm': array of pairwise distances between 
          boundaries in the vector form (as returned by 
          scipy.spatial.distance.pdist()) according to the order given in 
          bound_ids

          See calculateProperties() for other properties that are calculated
          and set in the 
          """

        if catalog is not None:
            clust = cls._readCatalog(
                files=files, mode=mode, catalog=catalog, categories=categories, 
                order=order, distances=distances)
        else:
            clust = cls._readOld(
                files=files, mode=mode, categories=categories, 
                order=order, distances=distances)

        return clust

    @classmethod
    def _readOld(cls, files, mode, categories=None, order=None, distances=None):
        """
        Does the job of read() for old style meta-data (not based on catalog).
        """

        # initialize
        db = pyto.io.Pickled(files)
        clusters = cls()

        # use all categories if not specified
        if categories is None:
            categories = list(db.categories())
            
        # set properties to be read
        if (mode == 'conn') or (mode == 'connectivity'):
            properties = ['connectivityBoundaries.clusters',
                          'connectivityBoundaries.nClusters',
                          'connectivityBoundaries.dataIds',
                          'connectivityBoundaries.nConnections',
                          'connectivityBoundaries.nLinks',
                          'connectivityBoundaries.euler',
                          'connectivityBoundaries.nLoops',
                          'connectivityBoundaries.eulerLinks',
                          'connectivityBoundaries.nLoopsLinks',
                          'connectivityConnections.clusters']
        elif (mode == 'hiBound') or (mode == 'hierarchicalBoundaries'):
            properties = ['hierarchyBoundaries.clusters',
                          'hierarchyBoundaries.nClusters',
                          'hierarchyBoundaries.dataIds',
                          'dualHierarchyConnections.clusters']
        elif (mode == 'hiConn') or (mode == 'hierarchicalConnections'):
            properties = ['dualHierarchyBoundaries.clusters',
                          'dualHierarchyBoundaries.nClusters',
                          'dualHierarchyBoundaries.dataIds',
                          'hierarchyConnections.clusters']
                                    
        # read property values for categories
        for categ in categories:

            # check if data for the current category exist 
            logging.debug('Clusters: Reading group ' + categ)
            if categ not in list(db.categories()):
                logging.warning(
                    'Clusters: Data for group ' + categ + ' do not exist')

            # make sure the identifier order is the same
            if order is not None:
                if isinstance(order[categ], Observations):
                    identifier = order[categ].identifiers
                elif isinstance(order[categ], (list, tuple)):
                    identifier = order[categ]
            else:
                identifier = None
            
            # check if requested identifiers exist in the database
            if identifier is not None:
                clean = []
                for requested in identifier:
                    if requested in db.identifiers():
                        clean.append(requested)
                    else:
                        logging.warning(
                            'CleftRegions: Data for experiment ' + requested + 
                            ' do not exist')
                identifier = clean

            # read data
            clusters[categ] = \
                db.readProperties(category=categ, identifier=identifier, 
                           index=None, properties=properties, compactify=False) 
                                    
        # rename properties
        if (mode == 'conn') or (mode == 'connectivity'):
            for categ in categories:
                clusters[categ].bound_clusters = \
                    clusters[categ].connectivityBoundaries_clusters 
                clusters[categ].n = \
                    clusters[categ].connectivityBoundaries_nClusters 
                clusters[categ].bound_ids = \
                    clusters[categ].connectivityBoundaries_dataIds 
                clusters[categ].n_connections = \
                    clusters[categ].connectivityBoundaries_nConnections 
                clusters[categ].n_links = \
                    clusters[categ].connectivityBoundaries_nLinks 
                clusters[categ].euler = \
                    clusters[categ].connectivityBoundaries_euler 
                clusters[categ].loops = \
                    clusters[categ].connectivityBoundaries_nLoops 
                clusters[categ].euler_links = \
                    clusters[categ].connectivityBoundaries_eulerLinks 
                clusters[categ].loops_links = \
                    clusters[categ].connectivityBoundaries_nLoopsLinks 
                clusters[categ].conn_clusters = \
                    clusters[categ].connectivityConnections_clusters
        elif (mode == 'hiBound') or (mode == 'hierarchicalBoundaries'):
            for categ in categories:
                clusters[categ].bound_clusters = \
                    clusters[categ].hierarchyBoundaries_clusters 
                clusters[categ].n = \
                    clusters[categ].hierarchyBoundaries_nClusters 
                clusters[categ].bound_ids = \
                    clusters[categ].hierarchyBoundaries_dataIds 
                clusters[categ].conn_clusters = \
                    clusters[categ].dualHierarchyConnections_clusters
        elif (mode == 'hiConn') or (mode == 'hierarchicalConnections'):
            for categ in categories:
                clusters[categ].bound_clusters = \
                    clusters[categ].dualHierarchyBoundaries_clusters 
                clusters[categ].n = \
                    clusters[categ].dualHierarchyBoundaries_nClusters 
                clusters[categ].bound_ids = \
                    clusters[categ].dualHierarchyBoundaries_dataIds 
                clusters[categ].conn_clusters = \
                    clusters[categ].hierarchyConnections_clusters

        # set ids and convert properties including removing index 0 (sum) in 
        # topology related
        for categ in list(clusters.values()):
            categ.ids = [numpy.arange(num) + 1 for num in categ.n]
            categ.bound_clusters = \
                [numpy.asarray(clust) for clust in categ.bound_clusters]
            categ.conn_clusters = \
                [numpy.asarray(clust) for clust in categ.conn_clusters]
            for name in ['n_connections', 'n_links', 'euler', 'loops', 
                         'euler_links', 'loops_links']:
                value = [val[1:] for val in getattr(categ, name)]
                setattr(categ, name, value)

        # set book-keeping attributes
        for categ in list(clusters.values()):
            categ.index = 'ids'
            categ.indexed = set(['ids', 'bound_clusters', 'conn_clusters'])
            categ.properties = set([
                    'identifiers', 'categories', 'n', 'ids',
                    'bound_ids', 'bound_clusters', 'conn_clusters'])
            if (mode == 'conn') or (mode == 'connectivity'):
                categ.indexed = categ.indexed.union([
                        'n_connections', 'n_links', 
                        'euler', 'euler_links', 'loops', 'loops_links'])
                categ.properties = categ.properties.union([
                        'n_connections', 'n_links', 
                        'euler', 'euler_links', 'loops', 'loops_links'])

        # set boundary and connection cluster size
        clusters.findNItems()

        # read boundary distances
        if distances is not None:
            clusters.addDistances(files=distances, clusters=files)

        return clusters

    @classmethod
    def _readCatalog(cls, files, catalog, mode, 
                     categories=None, order=None, distances=None):
        """
        Does the job of read() for catalog based meta-data.
        """

        # initialize
        db = pyto.io.Pickled(files)
        inst = cls()

        # use all categories if not specified
        if categories is None:
            categories = list(db.categories())
            
        # set properties to be read
        if (mode == 'conn') or (mode == 'connectivity'):
            inst._full_properties = {
                'connectivityBoundaries.ids' : 'ids',
                'connectivityBoundaries.clusters' : 'bound_clusters',
                'connectivityBoundaries.nClusters' : 'n',
                'connectivityBoundaries.dataIds' : 'bound_ids',
                'connectivityBoundaries.nConnections' : 'n_connections',
                'connectivityBoundaries.nLinks' : 'n_links',
                'connectivityBoundaries.euler' : 'euler',
                'connectivityBoundaries.nLoops' : 'loops',
                'connectivityBoundaries.branches' : 'branches',
                'connectivityBoundaries.eulerLinks' : 'euler_links',
                'connectivityBoundaries.nLoopsLinks' : 'loops_links',
                'connectivityBoundaries.branchesLinks' : 'branches_links',
                'connectivityConnections.clusters' : 'conn_clusters'}
            # 'bound_clusters' and 'conn_clusters' added to indexed later
            inst._full_indexed = [
                'connectivityBoundaries.ids',
                'connectivityBoundaries.nConnections',
                'connectivityBoundaries.nLinks',
                'connectivityBoundaries.euler',
                'connectivityBoundaries.nLoops',
                'connectivityBoundaries.branches',
                'connectivityBoundaries.eulerLinks',
                'connectivityBoundaries.nLoopsLinks',
                'connectivityBoundaries.branchesLinks']
            index_full_name = 'connectivityBoundaries.ids'

        elif (mode == 'hiBound') or (mode == 'hierarchicalBoundaries'):
            inst._full_properties = {
                'hierarchyBoundaries.ids' : 'ids',
                'hierarchyBoundaries.clusters' : 'bound_clusters',
                'hierarchyBoundaries.nClusters' : 'n',
                'hierarchyBoundaries.dataIds' : 'bound_ids',
                'dualHierarchyConnections.clusters' : 'conn_clusters'}
             # 'bound_clusters' and 'conn_clusters' added to indexed later
            inst._full_indexed = ['hierarchyBoundaries.ids']
            index_full_name = 'hierarchyBoundaries.ids'

        elif (mode == 'hiConn') or (mode == 'hierarchicalConnections'):
            inst._full_properties = {
                'hierarchyConnections.ids' : 'ids',
                'hierarchyConnections.clusters' : 'conn_clusters',
                'hierarchyConnections.nClusters' : 'n',
                'hierarchyConnections.dataIds' : 'bound_ids',
                'dualHierarchyBoundaries.clusters' : 'bound_clusters'}
            # 'bound_clusters' and 'conn_clusters' added to indexed later
            inst._full_indexed = ['hierarchyConnections.ids']
            index_full_name = 'hierarchyConnections.ids'

        else:
           raise ValueError(
               "Mode ", mode, " is not understood. Acceptable values are ",
               "'conn' ('connectivity'), 'hiBound' ('hierarchicalBoundaries')",
               " and 'hiConn' ('hierarchicalConnections').")

        # properties and indexed for all modes
        inst._properties = set(inst._full_properties.values())
        inst._indexed = set([inst._full_properties[full_indexed] 
                             for full_indexed in inst._full_indexed])

        # loop over categories
        props_found = {}
        for categ in categories:

            # check if data for the current category exist 
            logging.debug('Clusters: Reading group ' + categ)
            if categ not in list(db.categories()):
                logging.warning(
                    'Clusters: Data for group ' + categ + ' do not exist')

            # make sure the identifier order is the same
            if order is not None:
                if isinstance(order[categ], Observations):
                    identifier = order[categ].identifiers
                elif isinstance(order[categ], (list, tuple)):
                    identifier = order[categ]
            else:
                identifier = None

            # check if requested identifiers exist in the database
            if identifier is not None:
                clean = []
                for requested in identifier:
                    if requested in db.identifiers():
                        clean.append(requested)
                    else:
                        logging.warning(
                            'CleftRegions: Data for experiment ' + requested + 
                            ' do not exist')
                identifier = clean

            # get data from all experiments of this category
            group = Observations()
            for group, obj, categ_tmp, name_tmp in db.readPropertiesGen(
                category=categ, identifier=identifier, 
                properties=inst._full_properties, index=index_full_name, 
                indexed=inst._full_indexed, multi=group):

                logging.debug('Read data of experiment ' + name_tmp) 

                # do something, perhaps
                pass

            # bound_clusters and conn_clusters converted to indexed here
            group.bound_clusters = [numpy.asarray(clust) for clust 
                                    in group.bound_clusters]
            group.conn_clusters = [numpy.asarray(clust) for clust 
                                   in group.conn_clusters]
            inst._indexed.update(['bound_clusters', 'conn_clusters'])

            # add data for this category
            inst[categ] = group

            # set array properties to empty arrays for observations without ids
            for obs_index in range(len(inst[categ].identifiers)):
                if inst[categ].ids[obs_index] is None:
                    for name in inst._indexed:
                        value = getattr(inst[categ], name)
                        value[obs_index] = numpy.array([])

            # figure out if some properties were not found
            found = set()
            for name in inst._properties:
                value = getattr(group, name, None)
                if value is None:
                    continue
                if all([x is None for x in value]):
                    continue
                found.add(name)

            # set book-keeping attributes
            inst[categ].index = 'ids'
            inst[categ].indexed = inst._indexed.intersection(found)
            inst[categ].properties = inst._properties.intersection(found)
            # need to have identifiers in
            inst[categ].properties.update(['identifiers', 'categories'])

            # add properties from catalog 
            inst[categ].addCatalog(catalog=catalog)

        # read boundary distances
        if distances is not None:
            inst.addDistances(files=distances, catalog=catalog, clusters=files)

        # calculate additional data properties
        inst.calculateProperties(mode=mode)

        # convert to nm
        #sinst.convertToNm(catalog=catalog)

        # check if all groups have the same properties
        last = None
        for categ in categories:
            if last is not None:
                if inst[categ].properties != last:
                    raise ValueError("Groups have different properties")
            last = inst[categ].properties
        inst._indexed.intersection_update(last)
        inst._properties.intersection_update(last)
        
        return inst

    def addDistances(self, files, catalog, clusters=None):
        """
        Reads distances between boundaries and saves them as property 
        'bound_dist'.

        If arg files is 'default' the distance pickle names are derived from 
        cluster pickle names (arg clusters)  by substituting '_cluster.pkl' by 
        '_bound_distances.pkl').

        Arguments:
          - files: dictionary of boundary distance pickle file names, or
          'default' to derive distance pickle names from cluster pickle names 
          - clusters: dictionary of cluster distance pickle file names

        Sets property:
          - 'bound_dist': (list of ndarrays) pairwise distances between 
          boundaries in the vector form (as returned by 
          scipy.spatial.distance.pdist()) according to the order given in 
          bound_ids
        """

        # initialize
        if files == 'default':

            # convert all cluster pickle names to distance pickle names
            clusters = deepcopy(clusters)
            db = pyto.io.Pickled(clusters)
            for categ, ident_dict in db.files.items():
                for ident, file_name in db.files[categ].items():
                    dir, clust_base = os.path.split(file_name)
                    index = clust_base.rindex('_cluster.pkl')
                    dist_base =  clust_base[:index] + '_bound_distances.pkl'
                    db.files[categ][ident] = os.path.join(dir, dist_base)

        else:

            # distance pickle names given
            db = pyto.io.Pickled(files)

        # loop over categories
        for categ in self:
            pixel = catalog.pixel_size

            # make sure the identifier order is the same
            identifiers = self[categ].identifiers

            # read distances
            self[categ].bound_dist = [
                dist for dist, foo, foo 
                in db.data(category=categ, identifier=identifiers)]
            self[categ].properties.add('bound_dist')

            # convert to nm
            bd_nm = self[categ].pixels2nm(
                name='bound_dist', conversion=pixel[categ])
            self[categ].bound_dist_nm = bd_nm
            self[categ].properties.add('bound_dist_nm')


    ###############################################################
    #
    # Methods that remove items
    #
    ##############################################################

    def removeBoundaries(self, ids):
        """
        Removes boundaries with specified ids from flat clusters and removes
        empty clusters, but does not reassign items to clusters.

        Arguments:
          - ids: (Groups) boundary ids, have to have the same structure 
          (categories and observations) as this instance. The ids are read from
          property ids'
        """

        # remove boundary ids
        for categ in list(ids.keys()):

            # continue if current category not this instance
            if categ not in list(self.keys()):
                continue
            
            for obs_ind in range(len(ids[categ].ids)):

                # remove boundary ids from current observation
                clean = [one_ids.difference(ids[categ].ids[obs_ind]) \
                             for one_ids in self[categ].bound_clusters[obs_ind]]
                self[categ].bound_clusters[obs_ind] = numpy.array(clean)
        
        # remove empty boundary clusters
        self.removeEmpty(mode='boundary')

    def removeEmpty(self, mode='boundary'):
        """
        Removes empty boundary (mode 'boundary', or 'bound') or connection 
        (mode 'connection' or 'conn') clusters.
        """
 
        # set name of boundary / connection variable 
        if (mode == 'boundary') or (mode == 'bound'):
            name = 'bound_clusters'
        elif (mode == 'connection') or (mode == 'conn'):
            name = 'conn_clusters'
        else:
            raise ValueError ('Sorry, mode ' + mode + " was not understood. " \
                                  + "Mode can be 'boundary' or 'connection'.") 

        # remove 
        for categ in list(self.keys()):

            non_empty = []
            for obs_ind in range(len(self[categ].ids)):

                # make condition for current observation
                item_ids = getattr(self[categ], name)
                cond = [len(ids) > 0 for ids in item_ids[obs_ind]]
                non_empty.append(numpy.array(cond))

            # remove from current observations
            self[categ] = self[categ].extractIndexed(condition=non_empty)


    ###############################################################
    #
    # Calculation of other properties
    #
    ##############################################################

    def calculateProperties(self, mode):
        """
        Calculates additional properties:
          - number of items (see findNItems())
          - cluster fractions (see findBoundFract())
          - connections and links redundancy factors (see findRedundancy())

        See these methods for the properties that are assigned.

        Argument:
          - mode: clustering mode: 'connectivity' (or 'conn'), 
          'hierarchicalBoundaries' (or 'hiBound'), or 'hierarchicalConnections' 
          (or 'hiConn')

        """

        # calculate number of items 
        self.findNItems()

        # calculate cluster fractions
        self.findBoundFract()

        # calculate connections and links redundancy factors
        if (mode == 'conn') or (mode == 'connectivity'):
            self.findRedundancy()

    def findNItems(self, mode=None):
        """
        Calculates numbers of items (boundaries and connections) in each 
        cluster (mode 'in_cluster') or for each observation (mode 
        'in_observation'). Sets calculated numbers to self.n_bound_clust and 
        self.n_conn_clust (mode 'in_cluster') or to self.n_bound_obs and 
        self.n_conn_obs (mode 'in_observation').

        Argument:
          - mode: can be 'in_cluster' (same as 'in_clust'), 'in_observation' 
          ('in_obs') or None (both 'in_cluster' and 'in_observation' are 
          calculated.

        Sets properties:
          - n_bound_clust (indexed): (list of ndarrays) number of boundaries in 
          each (boundary) cluster, that is size of each boundary cluster
          - n_conn_clust (indexed): (list of ndarrays) number of connections in 
          each (connection) cluster, that is size of each connection cluster
          - n_bound_obs: (list) number of boundaries in each observation
          - n_conn_obs: (list) number of connections in each observation
        """

        # both modes
        if mode is None:
            self.findNItems(mode='in_cluster')
            self.findNItems(mode='in_obs')
            return

        for categ in self:

            # calculate number of boundaries for each observation, in each 
            # cluster 
            n_bound = []
            for bound in self[categ].bound_clusters:
                n_bound.append(numpy.array([len(clust) for clust in bound]))

            # calculate number of connections for each observation, in each 
            # cluster 
            n_conn = []
            for conn in self[categ].conn_clusters:
                n_conn.append(numpy.array([len(clust) for clust in conn]))

            # set data and book-keeping attributes
            if (mode == 'in_clust') or (mode == 'in_cluster'):
 
                self[categ].n_bound_clust = n_bound
                self[categ].n_conn_clust = n_conn

                self[categ].properties.add('n_bound_clust')
                self[categ].properties.add('n_conn_clust')
                self[categ].indexed.add('n_bound_clust')
                self[categ].indexed.add('n_conn_clust')

            elif (mode == 'in_obs') or (mode == 'in_observation'):

                self[categ].n_bound_obs = [nb_clust.sum() 
                                           for nb_clust in n_bound]
                self[categ].n_conn_obs = [nc_clust.sum() for nc_clust in n_conn]

                self[categ].properties.add('n_bound_obs')
                self[categ].properties.add('n_conn_obs')

            else:
                raise ValueError (
                    'Sorry, mode ' + mode + ' was not understood. '
                    + "Valid modes are 'in_cluster' and 'in_observation'." )

    def findBoundFract(self, categories=None):
        """
        For each boundary cluster calculates the fraction of total number of
        boundaries present (in that observation) that exist in the cluster.

        Arguments:
          - categories
        
        Sets properties:
          - fract_bound (indexed): (list of ndarrays) fraction of the total
          boundaries in a cluster
          - fract_bound_max: (list) max boundary fraction for each observation
        """

        # get categories
        if categories is None:
            categories = list(self.keys())

        # calculate
        for categ in categories:

            # calculate bound fractions for each observation
            fract = [nb_clust / float(nb_obs) for nb_clust, nb_obs
                     in zip(self[categ].n_bound_clust, self[categ].n_bound_obs)]
            self[categ].fract_bound = fract
            self[categ].fract_bound_max = [fract_obs.max() 
                                           for fract_obs in fract]
                
            # update book-keeping properties
            self[categ].properties.add('fract_bound')
            self[categ].indexed.add('fract_bound')
            self[categ].properties.add('fract_bound_max')  

    def findDistance(self, items, distance, mode=None, categories=None):
        """
        In each cluster, finds item that has min or max distance (depending on 
        the mode). The distances are read from attribute distance of 
        items[category]. 

        Arguments:
          - items: (Groups) items of the cluster, have to have indexed property
          named (arg) distance
          - distance: name of the distance property of items (typically 
          'meanDistance_nm' or 'minDistance_nm' if items is an Vesicles object)
          - mode: distance mode, 'min' or 'max'
          - categories: list of categories

        Sets property:
          - min_distance: (indexed) set if mode is min or None
          - max_distance: (indexed) set if mode is max or None
        """

        # do all if mode is None
        if mode is None:
            self.findDistance(items=items, distance=distance, mode='min')
            self.findDistance(items=items, distance=distance, mode='max')
            return

        # get distance function
        if mode == 'min':
            dist_func = min
            name = 'min_distance'
        elif mode == 'max':
            dist_func = max
            name = 'max_distance'
        else:
            raise ValueError("Argument mode can be None, 'min', or 'max'.")

        # get categories
        if categories is None:
            categories = list(self.keys())

        # calculate
        for categ in categories:

            # initialize property that will hold the distances
            setattr(self[categ], name, [])
            curr_distance = getattr(self[categ], name)

            # get variables for this categ
            item_ids = getattr(items[categ], items[categ].index)
            item_dist = getattr(items[categ], distance)

            for obs_ind in range(len(self[categ].identifiers)):
                
                # make dictionary of item distances
                dist_dict = dict(list(zip(item_ids[obs_ind], item_dist[obs_ind]))) 

                # find min distance for all clusters in this observation
                dist = [dist_func(dist_dict[item_id] 
                                  for item_id in cluster) 
                        for cluster in self[categ].bound_clusters[obs_ind]]
                curr_distance.append(numpy.asarray(dist))

            # book-keeping
            self[categ].properties.add(name)
            self[categ].indexed.add(name)
        
    def findRedundancy(self, categories=None, mode=None):
        """
        If mode is 'connections' calculates connection redundancy factor, 
        that is:
 
          redundancy = number of loops / number of connections 
        
        for each cluster separately and for each observation.

        If mode is 'links' calculates link redundancy factor, that is:
 
          redundancy = number of loops_links / number of links 
        
        for each cluster separately and for each observation.

        If mode is None calculates all of the above

        Arguments:
          - categories: categories
          - mode: calculation mode 'connections', 'links' or None (both)

        Sets properties:
          - redundancy (indexed): (list of ndarrays) number of redundant 
          connections for each cluster (if mode is None or 'connection')
          - redundancy_obs: (list) number of redundant connections for each 
          observation (if mode is None or 'connection')
          - redundancy_links (indexed): (list of ndarrays) number of redundant 
          links for each cluster (if mode is None or 'links')
          - redundancy_links obs: (list) number of redundant links for each 
          observation (if mode is None or 'links')
        """

        # deal with mode
        if mode is None:

            # mode is None, do all
            self.findRedundancy(categories=categories, mode='connections')
            self.findRedundancy(categories=categories, mode='links')
            return

        elif mode == 'connections':

            # connections mode
            conn_name = 'n_connections'
            loop_name = 'loops'
            red_name = 'redundancy'
            red_obs_name = 'redundancy_obs'

        elif mode == 'links':

            # links mode
            conn_name = 'n_links'
            loop_name = 'loops_links'
            red_name = 'redundancy_links'
            red_obs_name = 'redundancy_links_obs'

        else:
            raise ValueError("Argument mode not understood. Acceptable values "
                             + "are: None, 'connections' and 'links'.")
        
        # get categories
        if categories is None:
            categories = list(self.keys())

        # calculate
        for categ in categories:

            setattr(self[categ], red_name, [])
            setattr(self[categ], red_obs_name, [])
            conn_value = getattr(self[categ], conn_name)
            loop_value = getattr(self[categ], loop_name)
            for n_loop, n_conn in zip(loop_value, conn_value):

                # calculate redundancy for each cluster separately
                n_loop = numpy.asarray(n_loop)
                n_conn = numpy.asarray(n_conn, dtype='float')
                #n_conn = numpy.array([len(x) for x in conn], dtype='float')
                # just to avoid divide by zero warning
                n_conn_fixed = numpy.where(n_conn==0, -1, n_conn)
                red = numpy.where(n_conn==0, 0, n_loop / n_conn_fixed)

                # calculate redundancy for each observation
                try:
                    red_obs = n_loop.sum() / n_conn.sum()
                except ZeroDivisionError:
                    if n_loop == 0:
                        red = 0.
                    else:
                        raise ValueError("Number of loops is 0, but number of "
                                         + "connections is not 0.")

                # set the values
                red_value = getattr(self[categ], red_name)
                setattr(self[categ], red_name, red_value + [red])
                red_obs_value = getattr(self[categ], red_obs_name)
                setattr(self[categ], red_obs_name, red_obs_value + [red_obs])
                
            # book-keeping
            self[categ].properties.add(red_name)
            self[categ].indexed.add(red_name)
            self[categ].properties.add(red_obs_name)

    def findBranching(self, categories=None, mode=None):
        """
        Work in progress
        """

        # deal with mode
        if mode is None:

            # mode is None, do all
            self.findBranching(categories=categories, mode='connections')
            self.findBranching(categories=categories, mode='links')
            return

        elif mode == 'connections':

            # connections mode
            conn_name = 'n_connections'
            branch_name = 'branches'
            branch_fract_name = 'branches_fract'
            branch_obs_name = 'branches_obs'
            branch_fract_obs_name = 'branches_fract_obst'

        elif mode == 'links':

            # links mode
            conn_name = 'n_links'
            branch_name = 'branches_links'

        else:
            raise ValueError("Argument mode not understood. Acceptable values "
                             + "are: None, 'connections' and 'links'.")
        
        # get categories
        if categories is None:
            categories = list(self.keys())

        # calculate
        for categ in categories:
            for ident in self[categ].identifiers():

                # get values
                conn = self.getValue(identifier=ident, name=conn_name)
                branch = self.getValue(identifier=ident, name=branch_name)

                # make branching fractions
                conn_fix = numpy.where(conn > 0, conn, -1).astype('float')
                branch_fract = numpy.where(conn > 0, branch / conn_fix, 0)
                self.setValue(identifier=ident, name=branch_fract_name,
                              value=branch_fract, indexed=True)

                # make branching per observation
                branch_obs = branch.sum()
                self.setValue(identifier=ident, name=branch_obs_name,
                              value=branch_obs, indexed=False)

                # make branching fraction per observation
                branch_fract_obs = branch.sum() / conn.sum().astype('float')
                self.setValue(identifier=ident, name=branch_fract_obs_name,
                              value=branch_fract_obs, indexed=False)
