"""
Defines class Layers that holds data from one or more observations 
(experiments) divided (classified) in groups.

The observations are expected to be generated by scripts/layers.py.  

# Author: Vladan Lucic (Max Planck Institute for Biochemistry)
# $Id$
"""
from __future__ import unicode_literals
from __future__ import absolute_import
from __future__ import division
from builtins import zip
#from past.utils import old_div

__version__ = "$Revision$"


#import warnings
import logging
from copy import copy, deepcopy

import numpy
#import scipy

import pyto
from ..util import nested
from .observations import Observations
from .groups import Groups


class Layers(Groups):
    """
    """

    ###############################################################
    #
    # Initialization
    #
    ##############################################################

    def __init__(self):
        """
        Initializes attributes
        """

        # initialize super 
        super(Layers, self).__init__()

    ###############################################################
    #
    # Input
    #
    ##############################################################

    @classmethod
    def read(cls, files, pixel=None, catalog=None, categories=None, order=None):
        """
        Reads values form synaptic vesicle layers tables

        If arg catalog is specified, additional properties are read from the
        catalog. In this case roperty pixel_size has to be specified in
        the catalog. Otherwise, if arg catalog is None, arg pixel has to be
        given.

        Argument files has to be a dictionary of dictionaries, where ouside
        keys are group names, inside keys experiment identifiers and 
        inside values file names. For example:

        files = {'group_a' : {'exp_1' : file_1,
                              'exp_2' : file_2,
                              ...             },
                 'group_b' : {'exp_5' : file_5,
                              ...             },
                 ...                           }

        Argument catalog has to be a Catalog object where the groups are 
        already formed (by using Catalog.makeGroups(), for example). That is,
        catalog has to contain the data in attributes that are themselves
        of type dict. For example:

        catalog.pixel_size = {'group_a' : {'exp_1' : pix_size_1,
                                           'exp_2' : pix_size_2,
                                            ...             },
                              'group_b' : {'exp_5' : pix_size_5,
                                            ...             },
                              ...                               }

        Args files and catalog have to have the same groups and observations.

        A category specified by arg categories, or an experiment 
        identifier specified by arg order that does not exist in the data
        (arg files) is ignored and a warning is generated. This condition
        often generates an exception at a later point.

        Arguments:
          - files: dictionary of layers table files
          - pixel: dictionary of pixel sizes
          - catalog: (Catalog) data about experiments
          - categories: list of categories
          - order: another Groups instance (or just a dictionary with group 
          names as keys and identifier lists as values) that defines the 
        order of the identifiers here

        Sets properties:
          - 'ids' (indexed): (list of ndarrays) layer ids
          - 'identifiers' 
          - 'occupancy' (indexed): fraction of a layer occupied by vesicles 
          - 'distance' (indexed): distance to the "layer 0" (active zone) in pix
          - 'distance_nm' (indexed): distance to the "layer 0" (active zone) 
          in nm
          - 'volume' (indexed): volume of each layer in pixels
          - 'volume_nm' (indexed): volume of each layer in nm^3
          """

        # initialize
        db = pyto.io.Table(files, profile='layers')
        layers = cls()

        # use all categories if not specified
        if categories is None:
            categories = list(db.categories())

        # read property values for categories
        for categ in categories:

            # check if data for the current category exist 
            logging.debug('Layers: Reading group ' + categ)
            if categ not in list(db.categories()):
                logging.warning(
                    'Layers: Data for group ' + categ + ' do not exist')

            # make sure the identifier order is the same
            if order is not None:
                if isinstance(order[categ], Observations):
                    identifier = order[categ].identifiers
                elif isinstance(order[categ], (list, tuple)):
                    identifier = order[categ]
            else:
                identifier = None
            
            # check if requested identifiers exist in the database
            if identifier is not None:
                clean = []
                for requested in identifier:
                    if requested in db.identifiers():
                        clean.append(requested)
                    else:
                        logging.warning(
                            'Layers: Data for experiment ' + requested + 
                            ' do not exist')
                identifier = clean

            # read data
            layers[categ] = \
                db.readProperties(category=categ, identifier=identifier,
                          properties=['ids', 'volume', 'occupancy']) 
            layers[categ].distance = copy(layers[categ].ids)

            # convert to nm in case arg catalog is not specified
            if catalog is None:
                layers[categ].distance_nm = \
                    layers[categ].pixels2nm(name='distance', 
                                            conversion=pixel[categ])
                layers[categ].volume_nm = \
                    layers[categ].pixels2nm(name='volume', 
                                            conversion=pixel[categ], power=3)
                layers[categ].surface_nm = \
                    layers[categ].pixels2nm(name='volume', 
                                            conversion=pixel[categ], power=2)

            # book-keeping
            layers[categ].index = 'ids'
            layers[categ].properties.add('distance')
            layers[categ].properties.add('distance_nm')
            layers[categ].properties.add('volume_nm')
            layers[categ].indexed = set(['ids', 'distance', 'distance_nm', 
                                         'volume', 'volume_nm', 'occupancy'])

            # add properties from catalog 
            if catalog is not None:
                layers[categ].addCatalog(catalog=catalog)

        # convert to nm in case arg catalog is given
        if catalog is not None:
            layers.convertToNm(catalog=catalog)

        # calculate occupied
        layers.getOccupied()

        return layers


    ###############################################################
    #
    # Methods to calculate other properties
    #
    ##############################################################

    def getOccupied(self):
        """
        Calculates number of voxels that are occupied by segments for each layer
        and saves it in self.occupied.
        """

        for categ in self:
            
            # calculate occupancy
            self[categ].occupied = \
                [vol * frac for vol, frac in zip(self[categ].volume, 
                                                 self[categ].occupancy)]

            # update properties
            self[categ].properties.add('occupied')
            self[categ].indexed.add('occupied')

    def convertToNm(self, catalog, categories=None):
        """
        Converts certain properties from pixels to nm. The new values are 
        assigned to (new) properties named by adding '_nm' to the 
        corresponding original property name.

        Currently converted properties are: distance, volume, surface

        Used only when arg catalog is passed to read() method. The catalog has
        to have 'pixel_size' defined.

        Arguments:
          - catalog: (Catalog) has to contain attribute pixel_size
          - categories: list of categories
        """

        if categories is None:
            categories = list(self.keys())

        for categ in categories:
            pixel = catalog.pixel_size

            # convert distance
            try:
                self[categ].distance_nm = self[categ].pixels2nm(
                    name='distance', conversion=pixel[categ])
                self[categ].properties.update(['distance_nm'])
                self[categ].indexed.update(['distance_nm']) 
            except TypeError:
                if ((self[categ].distance is None) 
                    or any(value is None for value in self[categ].distance)):
                    pass
                else:
                    raise

            # convert volume (to nm^3)
            try:
                self[categ].volume_nm = self[categ].pixels2nm(
                    name='volume', conversion=pixel[categ], power=3)
                self[categ].properties.update(['volume_nm'])
                self[categ].indexed.update(['volume_nm']) 
            except TypeError:
                if ((self[categ].volume is None) 
                    or any(value is None for value in self[categ].volume)):
                    pass
                else:
                    raise

            # convert volume as surface (to nm^2)
            try:
                self[categ].surface_nm = self[categ].pixels2nm(
                    name='volume', conversion=pixel[categ], power=2)
                self[categ].properties.update(['surface_nm'])
                self[categ].indexed.update(['surface_nm']) 
            except TypeError:
                if ((self[categ].volume is None) 
                    or any(value is None for value in self[categ].volume)):
                    pass
                else:
                    raise


    ###############################################################
    #
    # Other methods
    #
    ##############################################################

    def rebin(self, bins, pixel=None, categories=None):
        """
        Rebins the data of all observations according to the args bin and pixel.
        
        If arg pixel is not specified, all indexed properties (arrays) are
        binned according to the position of the array elements, For example,
        if arg bin = [0, 2, 4, 6]

        a volume array

          [1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14]

       is binned to:

         [ 3, 7, 11]

       Alternatively, if arg pixel is specified, it is assumed that each 
       position in the data array is 1 pixel thick and that bins are given
       if the length units. That is, in the above example with pixels
       having length of 0.5, the result is:

         [ 10, 26 ]
          
       When bins are not integers, trapesoidal rule is used to calculate
       rebinned data.

       Properies 'volume', 'volume_nm', 'surface_nm' and 'occupied' are 
       calculated in the above menner because they are additive. However, 
       property 'occupancy' is calculed from the binned 'occupied' and
       'volume'.

        Propeerty ids, is adjusted to the number of bins. Properties 'distance'
        and 'distance_nm' are adjusted accordingly. Non-indexed properties are 
        simply copied.

        Arguments:
          - bins: new bins in nm, or in pixels in pixel is None
          - pixel: pixel size in nm (dictionary of pixel sizes, such as
          {categ_1 : {'id_1':pixel, 'id_2':pixel}, categ_2 {'id_3':pixel}}) 
        """

        # set categories
        if categories is None:
            categories = self

        # initialize
        rebinned = deepcopy(self)
        bins = numpy.asarray(bins, dtype='float')

        # calculate properties 
        for categ in categories:

            # convert bins from nm to pixels
            if pixel is None:
                conversion = numpy.ones(len(self[categ].identifiers))
            else:
                conversion = numpy.array(\
                    [pixel[categ][ident] for ident in self[categ].identifiers])
            pix_bins = [bins / c_one for c_one in conversion] 
            
            # rebin volume and occupied
            rebinned[categ].volume = self[categ].rebin(name='volume', 
                                                       bins=pix_bins)
            rebinned[categ].volume_nm = self[categ].rebin(name='volume_nm', 
                                                          bins=pix_bins)
            rebinned[categ].surface_nm = self[categ].rebin(name='surface_nm', 
                                                          bins=pix_bins)
            rebinned[categ].occupied = self[categ].rebin(name='occupied', 
                                                         bins=pix_bins)

            # calculate occupancy
            rebinned[categ].occupancy = [
                occ.astype('float') / vol for occ, vol 
                in zip(rebinned[categ].occupied, rebinned[categ].volume)]

            # set ids, distance and distance_nm
            rebinned[categ].ids = []
            rebinned[categ].distance = []
            rebinned[categ].distance_nm = []
            for pb in pix_bins:
                rebinned[categ].ids.append(numpy.arange(1,len(pb)))
                rebinned[categ].distance.append(copy(pb[:-1]))
                rebinned[categ].distance_nm.append(copy(bins[:-1]))

            # set properties
            rebinned[categ].properties = copy(self[categ].properties)

            # set other properties
            for name in rebinned[categ].properties:
                if name not in self[categ].indexed:

                    # copy non-indexed properties
                    for ident in self[categ].identifiers:
                        old_nonind = self[categ].getValue(
                            property=name, identifier=ident)
                        rebinned[categ].setValue(
                            property=name, identifier=ident, value=old_nonind)
                elif name not in [
                    'ids', 'identifiers', 'volume', 'volume_nm', 'surface_nm', 
                    'occupancy', 'occupied', 'distance', 'distance_nm']:

                    # warn if other properties are present
                    logging.warn("Don't know how to rebin property " + name \
                                     + ". Leaving it not rebinned.") 

        # calculate occupied
        rebinned.getOccupied()

        return rebinned
