"""
Defines class Vesicles that holds data from one or more observations 
(experiments) divided (classified) in groups.

The observations are expected to be generated by scripts/vesicles.py.  

# Author: Vladan Lucic (Max Planck Institute for Biochemistry)
# $Id$
"""
from __future__ import unicode_literals
from __future__ import absolute_import
from builtins import zip
from builtins import range
from past.builtins import basestring

__version__ = "$Revision$"


import warnings
import logging
from copy import copy, deepcopy

import numpy
import scipy

import pyto
from ..util import nested
from .observations import Observations
from .experiment import Experiment
from .groups import Groups


class Vesicles(Groups):
    """
    """

    ###############################################################
    #
    # Initialization
    #
    ##############################################################

    def __init__(self):
        """
        Initializes attributes
        """

        # initialize super 
        super(Vesicles, self).__init__()


    ###############################################################
    #
    # Input
    #
    ##############################################################

    @classmethod
    def read(cls, files, catalog=None, pixel=None, categories=None, 
             order=None, membrane=None, lumen=None, additional=[]):
        """
        Reads values from vesicle pickles.

        If arg catalog is specified, properties defined in the catalog are 
        also read. In this case roperty pixel_size has to be specified in
        the catalog. Otherwise, if arg catalog is None, arg pixel has to be
        given.

        Argument files has to be a dictionary of dictionaries, where ouside
        keys are group names, inside keys experiment identifiers and 
        inside values file names. For example:

        files = {'group_a' : {'exp_1' : file_1,
                              'exp_2' : file_2,
                              ...             },
                 'group_b' : {'exp_5' : file_5,
                              ...             },
                 ...                           }

        Argument catalog has to be a Catalog object where the groups are 
        already formed (by using Catalog.makeGroups(), for example). That is,
        catalog has to contain the data in attributes that are themselves
        of type dict. For example:

        catalog.pixel_size = {'group_a' : {'exp_1' : pix_size_1,
                                           'exp_2' : pix_size_2,
                                            ...             },
                              'group_b' : {'exp_5' : pix_size_5,
                                            ...             },
                              ...                               }

        Args files and catalog have to have the same groups and observations.

        A category specified by arg categories, or an experiment 
        identifier specified by arg order that does not exist in the data
        (arg files) is ignored and a warning is generated. This condition
        often generates an exception at a later point.

        By default the following attributes are read from the vesicle pickle:
        'vesicleIds', 'density.mean', 'minDistance', 'meanDistance' and 
        'mor.radius.mean'. These are saved in this object as properties:
        'ids', 'density', 'minDistance', 'meanDistance' and 'radius'. Also,
        property 'diameter_edge' is calculated from 'radius'. 
                 
        Furthermore, poperties expressed in pixels are converted to 
        nm ('minDistance_nm', 'meanDistance_nm', 'radius_nm' and
        'diameter_edge_nm').
        
        To read other attributes of the vesicle pickle object, they need 
        to be specified in arg additional. These attributes have to be simple,
        that is not of the form object.attribute. Attributes 'maxDistance'
        and 'centerDistance' are also converted to nm.

        Arguments:
          - files: dictionary of vesicle result pickle files
          - pixel: dictionary of pixel sizes
          - categories: list of categories
          - order: another Groups instance (or just a dictionary with group 
          names as keys and identifier lists as values) that defines the 
          order of the identifiers here
          - membrane: dictionary of vesicle membrane result pickle files 
          - lumen: dictionary of vesicle lumen result pickle files 
          - additional (list): elements of properties that should be read
          in addition to those listed below

        Sets properties:
          - 'ids' (indexed): (list of ndarrays) vesicle ids
          - 'identifiers' 
          - 'categories'
          - 'minDistance', 'meanDistance', 'minDistance_nm', 'meanDistance_nm'
          (all indexed): min and mean distance in pixels and nm to a given
          region
          - 'radius', 'radius_nm' (indexed): radius, that is the mean distance
          between vesicle center and surface elements
          - 'diameter_edge', 'diameter_edge_nm' (indexed): diameter from edge 
          to edge, calculated as 2 * radius + 1
          - 'density' (indexed): mean grey-scale density
          """

        # initialize
        db = pyto.io.Pickled(files)
        vesicles = cls()

        # use all categories if not specified
        if categories is None:
            categories = list(db.categories())

        # loop over categories
        for categ in categories:

            # check if data for the current category exist 
            logging.debug('Vesicles: Reading group ' + categ)
            if categ not in list(db.categories()):
                logging.warning(
                    'Vesicles: Data for group ' + categ + ' do not exist')

            # make sure the identifier order is the same
            if order is not None:
                if isinstance(order[categ], Observations):
                    identifier = order[categ].identifiers
                elif isinstance(order[categ], (list, tuple)):
                    identifier = order[categ]
            else:
                identifier = None
            
            # check if requested identifiers exist in the database
            if identifier is not None:
                clean = []
                for requested in identifier:
                    if requested in db.identifiers():
                        clean.append(requested)
                    else:
                        logging.warning(
                            'Vesicles: Data for experiment ' + requested + 
                            ' do not exist')
                identifier = clean

            # get data
            properties_to_read=[
                'vesicleIds', 'density.mean', 'minDistance', 'meanDistance', 
                'mor.radius.mean'] + additional
            vesicles[categ] = db.readProperties(
                category=categ, identifier=identifier,
                properties=properties_to_read, index='vesicleIds')
            vesicles[categ].ids = vesicles[categ].vesicleIds
            vesicles[categ].density = vesicles[categ].density_mean
            vesicles[categ].radius = vesicles[categ].mor_radius_mean
            vesicles[categ].diameter_edge = \
                [2 * rad + 1 for rad in vesicles[categ].mor_radius_mean]

            # convert to nm in case arg catalog is not specified
            if catalog is None:
                vesicles[categ].minDistance_nm = vesicles[categ].pixels2nm(
                    name='minDistance', conversion=pixel[categ])
                vesicles[categ].meanDistance_nm = vesicles[categ].pixels2nm(
                    name='meanDistance', conversion=pixel[categ])
                vesicles[categ].radius_nm = vesicles[categ].pixels2nm(
                    name='radius', conversion=pixel[categ])
                vesicles[categ].diameter_edge_nm = vesicles[categ].pixels2nm(
                    name='diameter_edge', conversion=pixel[categ])

            # set book-keeping attributes
            group = vesicles[categ]
            group.index = 'ids'
            indexed_all = set(
                ['ids', 'minDistance', 'meanDistance', 'radius', 
                 'diameter_edge', 'density'] 
                + additional)
            properties_all = indexed_all.copy()
            properties_all.update(set(['identifiers', 'categories']))
            properties = set([
                name for name in properties_all 
                if ((getattr(group, name, None) is not None) and 
                    not all([x is None for x in getattr(group, name, None)]))])
            indexed = set([
                name for name in indexed_all 
                if ((getattr(group, name, None) is not None) and 
                    not all([x is None for x in getattr(group, name, None)]))])

            # overwrite previous group.properties because properties contain 
            # updated names
            group.properties = properties
            group.indexed = indexed

            # add properties from catalog 
            if catalog is not None:
                vesicles[categ].addCatalog(catalog=catalog)

        # convert to nm in case arg catalog is given
        if catalog is not None:
            vesicles.convertToNm(catalog=catalog)

        # add density 
        if membrane is not None:
            vesicles.addDensity(files=membrane, prefix='membrane_')
        if lumen is not None:
            vesicles.addDensity(files=lumen, prefix='lumen_')

        # calculate other properties
        vesicles.calculateProperties(categories=categories)

        return vesicles

    def addDensity(self, files, prefix=''):
        """
        Reads mean grayscale density for each vesicle from (arg) files and ads
        the densities as a property named (arg) prefix + density.

        Meant for lumen and membrane density.

        Arguments:
          - files: dictionary of vesicle result pickle files
          - prefix: density property name prefix
        """

        # initialize
        db = pyto.io.Pickled(files)

        # loop over categories
        for categ in self:

            # make sure the identifier order is the same
            identifiers = self[categ].identifiers

            # get data
            props = db.readProperties(category=categ, identifier=identifiers,
                          properties=['density.mean'])
            name = prefix + 'density'
            setattr(self[categ], name, props.density_mean) 

            # adjuste property lists
            self[categ].properties.add(name)
            self[categ].indexed.add(name)

    def addLinked(self, files, contactName='contacts'):
        """
        Reads Contact objects from pickles and extract ids of vesicles that
        are linked to each vesicle. Saves this info in linked property.

        Attribute:
          - files: dictionary of (connections) result pickle files that 
          contain Contact object
          - name: attribute name of pickled objects that contains Contact
        """

        # initialize pickle read
        db = pyto.io.Pickled(files)

        # initialize lists to hold data
        for categ in list(self.values()):
            categ.linked = [None] * len(categ.identifiers)

        # loop over categories
        for obj, categ, ident in db.data(category=list(self.keys())):

            # get Contact object and recast it to the current contacts class 
            try:
                contact = getattr(obj, contactName)
            except AttributeError:
                contact = getattr(obj.labels, contactName)
            if contact.compactified:
                contact.expand()
            contact = pyto.segmentation.Contact.recast(contact)

            # find index for the current observation
            for obs_ind, loc_ident in zip(list(range(len(self[categ].identifiers))),
                                            self[categ].identifiers):
                if loc_ident == ident:
                    break

            # find ids of linked segments
            all_ids = self[categ].ids[obs_ind]
            linked_ids = contact.findLinkedBoundaries(ids=all_ids, 
                                                      distance=1, mode='exact')

            # make sure that only ids in all_ids are used
            linked_ids = [numpy.intersect1d(l_ids, all_ids) \
                              for l_ids in linked_ids]

            # convert to ndarray with object dtype
            linked_ids_nd = numpy.array([None]*len(linked_ids), dtype='object')
            for ves_ind in range(len(linked_ids)):
                linked_ids_nd[ves_ind] = linked_ids[ves_ind]

            # set linked attribute
            self[categ].linked[obs_ind] = linked_ids_nd

        # adjust property lists
        for categ in self:
            self[categ].properties.add('linked')
            self[categ].indexed.add('linked')


    ###############################################################
    #
    # Methods to calculate other properties
    #
    ##############################################################

    def convertToNm(self, catalog, categories=None):
        """
        Converts certain properties from pixels to nm. The new values are 
        assigned to (new) properties named by adding '_nm' to the 
        corresponding original property name.

        Currently converted properties are: minDistance, meanDistance,
        radius, diameter_edge.

        Used only when arg catalog is passed to read() method. The catalog has
        to have 'pixel_size' defined.

        Arguments:
          - catalog: (Catalog) has to contain attribute pixel_size
          - categories: list of categories
        """

        if categories is None:
            categories = list(self.keys())

        for categ in categories:
            pixel = catalog.pixel_size

            # convert minDistance
            try:
                self[categ].minDistance_nm = self[categ].pixels2nm(
                    name='minDistance', conversion=pixel[categ])
                self[categ].properties.update(['minDistance_nm'])
                self[categ].indexed.update(['minDistance_nm']) 
            except TypeError:
                if ((self[categ].minDistance is None) 
                    or any(value is None for value in self[categ].minDistance)):
                    pass
                else:
                    raise
        
            # convert maxDistance
            if 'maxDistance' in self[categ].properties:
                try:
                    self[categ].maxDistance_nm = self[categ].pixels2nm(
                        name='maxDistance', conversion=pixel[categ])
                    self[categ].properties.update(['maxDistance_nm'])
                    self[categ].indexed.update(['maxDistance_nm']) 
                except TypeError:
                    if ((self[categ].maxDistance is None) 
                        or any(value is None for value 
                               in self[categ].maxDistance)):
                        pass
                    else:
                        raise
        
            # convert centerDistance
            if 'centerDistance' in self[categ].properties:
                try:
                    self[categ].centerDistance_nm = self[categ].pixels2nm(
                        name='centerDistance', conversion=pixel[categ])
                    self[categ].properties.update(['centerDistance_nm'])
                    self[categ].indexed.update(['centerDistance_nm']) 
                except TypeError:
                    if ((self[categ].centerDistance is None) 
                        or any(value is None for value 
                               in self[categ].centerDistance)):
                        pass
                    else:
                        raise
        
            # convert meanDistance
            try:
                self[categ].meanDistance_nm = self[categ].pixels2nm(
                    name='meanDistance', conversion=pixel[categ])
                self[categ].properties.update(['meanDistance_nm'])
                self[categ].indexed.update(['meanDistance_nm']) 
            except TypeError:
                if ((self[categ].meanDistance is None) or
                    any(value is None for value in self[categ].meanDistance)):
                    pass
                else:
                    raise
        
            # convert radius
            try:
                self[categ].radius_nm = self[categ].pixels2nm(
                    name='radius', conversion=pixel[categ])
                self[categ].properties.update(['radius_nm'])
                self[categ].indexed.update(['radius_nm']) 
            except TypeError:
                if ((self[categ].radius is None) 
                    or any(value is None for value in self[categ].radius)):
                    pass
                else:
                    raise
        
            # convert diameter_edge
            try:
                self[categ].diameter_edge_nm = self[categ].pixels2nm(
                    name='diameter_edge', conversion=pixel[categ])
                self[categ].properties.update(['diameter_edge_nm'])
                self[categ].indexed.update(['diameter_edge_nm']) 
            except TypeError:
                if ((self[categ].diameter_edge is None) or
                    any(value is None for value in self[categ].diameter_edge)):
                    pass
                else:
                    raise
        
    def calculateProperties(self, categories=None):
        """
        Calculates additonal properties from the already existing properties. 

        Sets following new properties to each group (Observations instance)
        contained in this object:
          - lum_mem_density_diff: (indexed) difference between lumenal and
          membrane densities

        Argument:
          - categories: list of group names, in None all groups are used
        """

        if categories is None:
            categories = list(self.keys())

        # check if lumen and membrane exist
        do_lummem = True
        if categories is None:
            categories = list(self.keys())
        for g_name in categories:
            if (('lumen_density' not in self[g_name].properties) or
                ('membrane_density' not in self[g_name].properties)):
                do_lummem = False

        # calculate lumen - membrane density
        if do_lummem:
            self.apply(funct=numpy.subtract, 
                       args=('lumen_density', 'membrane_density'),
                       name='lum_mem_density_diff', categories=categories)

    def getNTethers(self, tether, name='n_tether', categories=None):
        """
        Calculats number of tethers for each vesicle.

        If name is None the calculated numbers are returned. Otherwise the
        numbers are assigned to each category (Observations) comprising this
        object as attribute specified by name.

        Arguments:
          - tether: (Connections) object containing tethers data
          - name: (string) name of the attribute of categories (Observations 
          objects) to which the calculated values are assigned.  
          - categories: list of group names, if None all groups are usedy
        """

        return self.getNConnections(conn=tether, name=name, 
                                    categories=categories)

    def getNConnections(self, conn, name='n_connection', categories=None):
        """
        Calculated number of connections for each vesicle.

        If name is None the calculated numbers are returned. Otherwise the
        numbers are assigned to each category (Observations) comprising this
        object as attribute specified by name.

        Arguments:
          - conn: (Connections) object containing connections data
          - name: (string) name of the attribute of categories (Observations 
          objects) to which the calculated values are assigned.  
          - categories: list of group names, if None all groups are usedy
        """

        # set categories
        if categories is None:
            categories = list(self.keys())

        # calculate number of connections for each vesicle
        n_conn = {}
        for categ in categories:

            # loop over observations
            n_conn[categ] = []
            for b_ids, ves_ids in zip(conn[categ].boundaries, self[categ].ids):

                b_ids_flat = numpy.asarray(pyto.util.nested.flatten(b_ids))
                n_conn_obs = [(b_ids_flat == v_id).sum() for v_id in ves_ids]
                n_conn[categ].append(numpy.asarray(n_conn_obs))

        # return or save
        if name is None:

            # return the calculated numbers
            return n_conn

        else:

            # assign the calculated numbers to an attribute 
            for categ in categories:
                setattr(self[categ], name, n_conn[categ])
                self[categ].properties.add(name)
                self[categ].indexed.add(name)
            
    def getNLinked(self, categories=None):
        """
        Sets property n_linked to the number of linked vesicles for each
        vesicle.
        """

        # set categories
        if categories is None:
            categories = list(self.keys())

        # calculate number of linked vesicles for each vesicle
        n_link = {}
        for categ in categories:

            # loop over observations
            self[categ].n_linked = []
            for obs in self[categ].linked:
                n_linked = numpy.array([len(ids) for ids in obs])
                self[categ].n_linked.append(n_linked)

            # adjust property lists
            self[categ].properties.add('n_linked')
            self[categ].indexed.add('n_linked')
    
    def getMeanConnectionLength(self, conn, categories=None, 
                                name='mean_connection_nm', value=-1):
        """
        Calculates mean connection length (in nm) for each sv and adds it as a 
        new property to this object.

        The (arg) conn object has to have properties 'boundaries' and 
        'length_nm'. Each connection has to have exactly two boundaries.

        Arguments:
          - conn: connections object
          - categories: list of categories, if None all categories are used
          - name: name of the property holding mean connection length (in nm)
          - value: default value for mean connection length in case no 
          connections found (indication of error)
          """
    
        # set categories
        if categories is None:
            categories = list(self.keys())

        # loop over groups and experiments
        for categ in categories:
            for ident in conn[categ].identifiers:
                sv_dict = {}

                # make sv id - conn lengths dictionary
                for bound, length in zip(
                    conn[categ].getValue(identifier=ident, 
                                           property='boundaries'), 
                    conn[categ].getValue(identifier=ident, 
                                           property='length_nm')):
                    try:
                        sv_dict[bound[0]].append(length)
                    except KeyError:
                        sv_dict[bound[0]] = [length]
                    try:
                        sv_dict[bound[1]].append(length)
                    except KeyError:
                        sv_dict[bound[1]] = [length]

                # make sv id : mean conn length dictionary
                sv_dict_mean = {}
                for ind, lengths in list(sv_dict.items()):
                    sv_dict_mean[ind] = numpy.array(lengths).mean()

                # add new property to sv
                new = [sv_dict_mean.get(id_, value) for id_ 
                       in self[categ].getValue(identifier=ident, 
                                               property='ids')]
                self[categ].setValue(identifier=ident, property=name, 
                                   value=numpy.array(new), indexed=True)

    def getNVesicles(
        self, name, categories=None, inverse=False, fixed=None, layer=None, 
        layer_name='surface_nm', layer_index=1, layer_factor=1.e-6):
        """
        Calculate number of vesicles or a related property (depending on the
        arguments) for each observation and saves it as a separate (non-indexed)
        property named by arg name.

        If arg layers in None, the number of vesicles is calculated.

        If arg layers is specified, number of vesicles per unit layer area
        is calculated. The unit layer area is calculated as:

          layer_name[layer_index] * layer_factor

        In a typical case layer area in nm (property 'surface_nm') of the
        layer 1 (layer_index=1). If layer_factor=1.e-6 (default), then this 
        method calculates number of vesicles per 1 um^2 of the layer surface.

        If arg inv is True, the inverse is calculated, that is the layer area 
        per one sv.

        If arg fixed is specified, te calculation proceed as above except that
        the number of vesicles (for each observation) is fixed to this number.
        For example, if fixed=1 and inverse=True, the area of the layer
        specifed by arg layer_index is calculated.

        Sets property named (arg) name.

        Arguments:
          - name: name of the newly calculated property
          - inverse: flag indicating if inverse of the property should be 
          calculated
          - layer: object containing layer info
          - layer_name: name of the layer property that contains the desired 
          layer data
          - layer_index: index of the layer whose surface is used
          - layer_factor: the surface is multiplied by this factor
        """

        self.getN(
            name=name, categories=categories, inverse=inverse, fixed=fixed, 
            layer=layer, layer_name=layer_name, layer_index=layer_index, 
            layer_factor=layer_factor)

    def getConnectivityDistance(
            self, initial, distance=1, name='conn_distance',
            categories=None, default=-1, max_distance=10**6):
        """
        Calculates connectivity distance for each vesicle.

        The connectivity distance is calculated as follows. First, all vesicles
        contained in object specified by arg initial are assigned the distance
        given by arg distance. For all other vesicle, the shortest path (the
        smallest number of connectors) to any of the initial vesicles is 
        calculated. This number is added to the arg distance to yield
        the connectivity distance.

        Specifically, if vesicles tethered to a boundary are given by arg 
        initial, these vesicles will have the connectivity distance 1 
        (provided that arg distance=1), the vesicles connected to the 
        tethered vesicles (but not tethered themselves) will have the distance
        2 and so on.

        The calculated distances are saved as an indexed property with name 
        given by arg name.

        Vesicles for which there is no connector / vesicle path to the
        initial vesicles are assigned the distance given by arg default.

        All groups of this object have to have attribute 'linked' set (see
        addLinked() method). Otherwise ValueError is raised.

        Arguments:
          - initial: instance of this class that contains vesicles for which
          the distance is given by arg distance
          - distance: connectivity distance of initial vesicles
          - name: name of the property used to store the connectivity distance
          values
          - categories: group names
          - default: distance value given to vesicles that are not linked to
          the initial vesicles
          - max_distance: max distance allowed (just to prevent infinite loop)
        """

        # set categories
        if categories is None:
            categories = list(self.keys())

        # loop over groups and experiments
        for categ in categories:

            # initialize connection distance to default
            self[categ].setValue(
                identifier=None, name=name, indexed=True, default=default)

            
            for ident in self[categ].identifiers:

                # set initial distance
                ids = self[categ].getValue(identifier=ident, name='ids')
                init_ids = initial[categ].getValue(identifier=ident, 
                                                    name='ids')
                for id_ in init_ids:
                    self[categ].setValue(
                        identifier=ident, name=name, value=distance,
                        id_=id_, indexed=True)

                if 'linked' not in self[categ].properties:
                    raise ValueError("Property 'linked' has to be set in order"
                                     + " to calculate connectivity distance.")

                # set other distances
                previous_ids = init_ids
                curr_distance = distance + 1
                while(True):
                #for curr_distance in range(distance+1, max_distance+1):

                    # loop over vesicles and set current distance
                    found = False
                    current_ids = []
                    ids = self[categ].getValue(identifier=ident, name='ids')
                    for id_ in previous_ids: 

                        # find linked to this vesicle
                        linked = self[categ].getValue(
                            identifier=ident, name='linked', ids=id_)
                        #print('ident: ', ident, ' dist: ', curr_distance, 
                        #      ' id_: ', id_, ' linked: ', linked)

                        # restrict to existing ids
                        linked = numpy.intersect1d(linked, ids)

                        # loop over linked, assign distance if has default dist 
                        for linked_id in linked:
                            linked_distance = self[categ].getValue(
                                    identifier=ident, name=name, ids=linked_id)
                            if linked_distance == default:
                                self[categ].setValue(
                                    identifier=ident, name=name, 
                                    value=curr_distance,
                                    id_=linked_id, indexed=True)
                                current_ids.append(linked_id)
                                found = True

                    # break if no more linked 
                    if not found: break

                    # error if max distance reached
                    if curr_distance == max_distance:
                        raise ValueError(
                            "Max connectivity distance reached, try increasing"
                            + " the max_distance argument.")

                    # prepare for the next iteration
                    previous_ids = current_ids
                    curr_distance =  curr_distance + 1

    def getClusterSize(self, clusters, categories=None):
        """
        For each vesicle finds cluster size of the cluster that it belongs to.

        Vesicles that appear in clusters but not in (self) ids are ignored, but
        all vesicles from ids have to exist in clusters.

        Arguments:
          - clusters: (Groups with Clusters values) vesicle clusters

        Sets property:
          - cluster_size (indexed)
        """
        # set categories
        if categories is None:
            categories = list(self.keys())

        # calculate cluster size for each vesicle
        for categ in categories:
            self[categ].cluster_size = []

            # loop over observations
            for obs_ind in range(len(clusters[categ].ids)):

                # make dictionary of cluster sizes for this observation
                sizes = {}
                for clust, clust_size \
                        in zip(clusters[categ].bound_clusters[obs_ind], 
                               clusters[categ].n_bound_clust[obs_ind]):
                    sizes.update(dict(list(zip(clust, [clust_size] * clust_size))))

                # make array of sizes in the same order as ids
                ordered = [sizes[id_] for id_ in self[categ].ids[obs_ind]]
                self[categ].cluster_size.append(numpy.array(ordered))

            # update property lists
            self[categ].properties.add('cluster_size')
            self[categ].indexed.add('cluster_size')

    def cluster(self, clusters, categories=None, method='average'):
        """
        Clusters vesicles for each observation and calculates cophenetic
        (correlation) coeficient (correlation between cluster hierarchy and 
        distances).

        Distances between vesicles are provided by arg clusters. Boundary ids of
        clusters have to be a subset of ids of this instance.

        Arguments:
          - clusters: (Clusters) property 'bound_dist' contains distances
          between 
          vesicles
          - categories
        
        Sets properties:
          - hi_cluster: (segmentation Cluster) hierarchical clusters
          - hi_cophen: cophenetic (correlation) coefficient 
        """

        # set categories
        if categories is None:
            categories = list(self.keys())

        # cluster svs and get cophenetic coef
        for categ in categories:
            self[categ].hi_cluster = []
            self[categ].hi_cophen = []
            for obs_ind in range(len(self[categ].identifiers)):

                # get distances for sv
                bound_ids = self[categ].ids[obs_ind]
                dist = pyto.segmentation.Cluster.extractDistances(\
                    distances=clusters[categ].bound_dist[obs_ind], 
                    ids=clusters[categ].bound_ids[obs_ind], keep=bound_ids)

                # cluster and get cophonetic coef
                if len(dist) > 0:
                    hi_clust = \
                        pyto.segmentation.Cluster.hierarchical(distances=dist, 
                                                 ids=bound_ids, method=method)
                    coef, coph_dist = \
                        scipy.cluster.hierarchy.cophenet(hi_clust.codeBook0, 
                                                         dist)

                else:
                    hi_clust = numpy.nan
                    coef = numpy.nan

                # set properties
                self[categ].hi_cluster.append(hi_clust)
                self[categ].hi_cophen.append(coef)

            # book-keeping
            self[categ].properties.add('hi_clust')
            self[categ].properties.add('hi_cophen')

    def getNearestNeighbor(
            self, cluster, dist_name='bound_dist', name='nearest', 
            categories=None, default=-1):
        """
        Finds the nearest neighbor (vesicle) id and distance for each 
        vesicles.

        The distances between all vesicles need to be contained in arg 
        cluster, with property names dist_name (in pixels) or 
        dist_name + '_nm' (in nm).

        Important: vesicle ids used to calculate distances between vesicles
        that are stored in Cluster have to be the same as the ids of this 
        instance (for all observations that comprise this instance). 

        Sets propertis of this instance:
          - name + '_distance': nearest neighbor distances in pixels, set if
          cluster contains distances in pixels
          - name + '_distance_nm': nearest neighbor distances in nm, set if
          cluster contains distances in nm
          - name + '_ids': nearest neighbor ids

        Arguments:
          - cluster: (pyto.segmentation.Cluster) cluster data that contain
          distances
          - dist_name: name of the cluster property where the distances are 
          saved
          - name: prefix for the names of distances and indices of nearest
          neighbors in this instance
          - categories: categories
          - default: value given to elements of the resulting properties that
          do not get nearest neighbor distances assigned (likely becuase
          distances between vesicles were not given)
        """

        # set categories
        if categories is None:
            categories = list(self.keys())

        # loop over categories and observations
        for categ in categories:
            for ident in self[categ].identifiers:
                bound_ids = numpy.asarray(
                    cluster[categ].getValue(identifier=ident, name='bound_ids'))
                self_ids = numpy.asarray(
                    self[categ].getValue(identifier=ident, name='ids'))

                # transform distances and find the nearest (pixels)
                dist = False
                if dist_name in cluster[categ].properties:
                    distances = numpy.asarray(cluster[categ].getValue(
                        identifier=ident, name=dist_name))
                    very_high = 100 * numpy.abs(distances.max() + 1)
                    distances_tr = Experiment.transformByIds(
                        ids=bound_ids, values=distances, new_ids=self_ids,
                        mode='vector_pair', default=very_high)
                    near_inds, near_dist = \
                        pyto.segmentation.Cluster.findClosest(
                            data=distances_tr, data_mode='distances_compact', 
                            ids=self_ids)
                    near_inds[near_dist == very_high] = default
                    near_dist[near_dist == very_high] = default
                    dist = True
                   
                # transform distances and find the nearest (nm)
                dist_name_nm = dist_name + '_nm'
                dist_nm = False
                if dist_name_nm in cluster[categ].properties:
                    distances = numpy.asarray(cluster[categ].getValue(
                        identifier=ident, name=dist_name_nm))
                    very_high = 100 * numpy.abs(distances.max() + 1)
                    distances_tr = Experiment.transformByIds(
                        ids=bound_ids, values=distances, new_ids=self_ids,
                        mode='vector_pair', default=very_high)
                    near_inds_nm, near_dist_nm = \
                        pyto.segmentation.Cluster.findClosest(
                            data=distances_tr, data_mode='distances_compact', 
                            ids=self_ids)
                    near_inds_nm[near_dist_nm == very_high] = default
                    near_dist_nm[near_dist_nm == very_high] = default
                    dist_nm = True
                   
                # make sure near_inds are set
                if dist_nm and not dist:
                    near_inds = near_inds_nm

                # set values
                if dist:
                    self[categ].setValue(
                        identifier=ident, name=name+'_distance', 
                        indexed=True, value=numpy.asarray(near_dist))
                if dist_nm:
                    self[categ].setValue(
                        identifier=ident, name=name+'_distance_nm', 
                        indexed=True, value=numpy.asarray(near_dist_nm))
                self[categ].setValue(
                    identifier=ident, name=name+'_ids', 
                    indexed=True, value=numpy.asarray(near_inds))


    ###############################################################
    #
    # Extraction methods
    #
    ##############################################################

    def splitByDistance(self, distance, name='meanDistance_nm', 
                        categories=None):
        """
        Returns a list of instances of this class, where each object contains 
        data for vesicles whose distances (attribute specified by name) fall
        into bins specified by arg distances.

        Lower distance bin limits are inclusive, while the upper are exclusive, 
        except for the upper limit of the last distance bin which is inclusive 
        (like numpy.histogram)

        If distance is a single number a single object is returned. Otherwise,
        if distance is a list of numbers, a list of objects is returned.

        Arguments:
          - distance: list of distances, interpreted as distance bins, or if 
          a single number it is a higher distance limit, while 0 is the lower
          - name: name of the distance attribute 
          - categories:
        """

        return self.split(value=distance, name=name, categories=categories)

    def splitByRadius(self, radius, name='radius_nm', categories=None):
        """
        Returns a list of instances of this class, where each object contains 
        data for vesicles whose radius (attribute specified by name) fall
        into bins specified by arg radius.

        Lower radius bin limits are inclusive, while the upper are exclusive, 
        except for the upper limit of the last radius bin which is inclusive 
        (like numpy.histogram)

        If radius is a single number a single object is returned. Otherwise,
        if radius is a list of numbers, a list of objects is returned.

        Arguments:
          - radius: list of radii, interpreted as radius bins, or if 
          a single number it is a higher radius limit, while 0 is the lower
          - name: name of the radius attribute 
          - categories:
        """

        return self.split(value=radius, name=name, categories=categories)

    def extractTethered(self, categories=None, name='n_tether', other=False):
        """
        Returns a new instance of this class that contains all tethered 
        vesicles (from all observations and for all categories).

        If other is True, an instance of this class containing all
        non-tethered vesicles is also returned.

        Property with name given by arg name has to be set and to show number of
        tethers for each vesicle.

        Arguments:
          - categories:
          - name: name of the property (attribute of this instance) that
          contains number of tethers for each vesicle
          - other: flag indicating if non-tethered vesicles are returned
        """
        
        return self.extractConnected(categories=categories, name=name,
                                     other=other)

    def extractConnected(self, categories=None, name='n_connection',
                         other=False):
        """
        Returns a new instance of this class that contains all connected 
        vesicles (from all observations and for all categories).

        If other is True, an instance of this class containing all
        non-connected vesicles is also returned.

        Property with name given by arg name has to be set and to show number of
        connections for each vesicle.

        Arguments:
          - categories:
          - name: name of the property (attribute of this instance) that
          contains
          number of connections for each vesicle
          - other: flag indicating if non-connected vesicles are returned
        """
        
        # get categories
        if categories is None:
            categories = list(self.keys())

        # check which elements are connected
        cond = {}
        other_cond = {}
        for categ in categories:
            cond[categ] = []
            other_cond[categ] = []
            for n_conn in getattr(self[categ], name):
                cond[categ].append(n_conn > 0)
                other_cond[categ].append(n_conn == 0)

        # extract connected 
        connected = self.extract(condition=cond)
        if other:
            non_connected = self.extract(condition=other_cond)

        # return
        if other:
            return connected, non_connected
        else:
            return connected
